/*
     eib.c - Low level EIB driver for BCU 1

     Copyright (C) 2000 Bernd Thallner <bernd@kangaroo.at>
     Copyright (C) 2005 Martin Kgler <mkoegler@auto.tuwien.ac.at>

     Ported to Linux 2.6 by Robert Kurevija
     Heavily modfied by Martin Kgler

     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define EIB_DEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/serial_reg.h>
#include <linux/interrupt.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/moduleparam.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#include"eib.h"

#if !defined(DEFAULT_MAJOR)
  #define DEFAULT_MAJOR 121
#endif

static int eib_major = DEFAULT_MAJOR;


MODULE_AUTHOR("Bernd Thallner, bernd@kangaroo.at");
MODULE_AUTHOR("Martin Kgler, mkoegler@auto.tuwien.ac.at");
MODULE_DESCRIPTION("A low level EIB driver for BCU 1. V0.2.6.1");
MODULE_PARM(eib_major, "i");
MODULE_PARM_DESC(eib_major, "Major number for this device.");




/* -- */
#define MAX_READ_BUFFER 20  /* * 33 byte buffer size */
#define MAX_WRITE_BUFFER 5  /* * 33 byte buffer size */


// enable debug messsages
#define EIB_DEBUG


#undef EIB_PRINT
#define EIB_PRINT(fmt, args...) printk(KERN_NOTICE "eib: " fmt "\n", ## args)

#undef PDEBUG
#ifdef EIB_DEBUG
#  define PDEBUG(fmt, args...) printk(KERN_NOTICE "eib: " fmt "\n", ## args)
#else
# define PDEBUG(fmt, args...) ;
#endif

// maximum time in usec to wait between rx interrupt raised and byte fully send
#define MAX_TEMT_WAIT 1000

// minimum time in usec to wait between byte fully send and clear rts signal
#define WRITE_TEMT_CLRRTS_DELAY 100

// minimum time in usec to wait between clear rts and set rts signal
#define WRITE_CLRRTS_SETRTS_DELAY 80

// timeout for data block transfer
#define READ_WRITE_TIMEOUT 130000

// maximum time between to interrupts if not in idle state
#define MAX_TIME_BETWEEN_INT 80000

// minimum time to wait after reset without any respond to signals
#define MIN_RESET_WAIT_TIME 130000

// maximum time between to wait for start sending
#define MAX_INIT_SEND_TIMEOUT 100000

//
#define MAX_INIT_SEND_TIME 3000


static spinlock_t open_lock = SPIN_LOCK_UNLOCKED;

/* private minor number use count */
static int used[4];


typedef struct {
  int minor;                   /* minor number */
  int port;
  int irq;
  spinlock_t lock;
  spinlock_t irqlock;
  struct tasklet_struct temt_tasklet;
  int shutdown;
/*
 * Save uart register settings for the serial port
 */
unsigned char old_dll;    /* dll = divisor latch low       */
unsigned char old_dlh;    /* dlh = divisor latch high      */
unsigned char old_lcr;    /* lcr = line control register   */
unsigned char old_mcr;    /* mcr = modem control register  */

struct timer_list initate_send_timer_list;
struct timer_list detect_reset_timer_list;

  wait_queue_head_t read_queue;
  int read_head, read_free, read_next;
  unsigned char read_buffer [MAX_READ_BUFFER][33];
  spinlock_t read_lock;
  int read_pos;

  wait_queue_head_t write_queue;
  int write_head, write_free, write_next;
  unsigned char write_buffer[MAX_WRITE_BUFFER][33];
  signed char write_result[MAX_WRITE_BUFFER];
  spinlock_t write_lock;
  int write_pos;

struct timespec last_int_tv;
struct timespec start_tv;
struct timespec send_tv;
struct timespec reset_tv;           /* Time of the last reset */
volatile int state;  /* actual state of the state-machine */
volatile int initate_send;
  unsigned long warning_flag;
  int reset_count;

} eib_t;


/*
 * Function headers
 *
 */
static int eib_open(struct inode *inode, struct file *file);
static int eib_release(struct inode *inode, struct file *file);
static void init_serial(eib_t *dev);
static void restore_serial(eib_t *dev);
static int bitcount(unsigned char x);
static inline long time_difference(struct timespec *tv1, struct timespec *tv2);
static void inline add_initate_send_timer(struct file* file);
static void inline add_detect_reset_timer(struct file* file);
static void initate_send_timer(unsigned long data);
static void detect_reset_timer(unsigned long data);
static irqreturn_t interrupt_handler(int irq, void *dev_id, struct pt_regs *regs);
static ssize_t eib_read(struct file *file, char __user *buf, size_t count, loff_t *offset);
static ssize_t eib_write(struct file *file, const char __user *buf, size_t count, loff_t *offset);
static unsigned int eib_poll(struct file *file, poll_table *wait);
static int eib_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int eib_flush(struct file *file);
static void do_temt_bh(unsigned long unused);     /* tasklet scheduled in the Interrupt handler */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11)
/* part of linux 2.6.11+ */
static inline unsigned long usecs_to_jiffies(const unsigned int u)
{
  if (u > jiffies_to_usecs(MAX_JIFFY_OFFSET))
    return MAX_JIFFY_OFFSET;
#if HZ <= 1000000 && !(1000000 % HZ)
  return (u + (1000000 / HZ) - 1) / (1000000 / HZ);
#elif HZ > 1000000 && !(HZ % 1000000)
  return u * (HZ / 1000000);
#else
 return (u * HZ + 999999) / 1000000;
#endif
}
#endif


static void inline add_initate_send_timer(struct file* file)  
{
  eib_t *dev = (eib_t*)file->private_data;
  long t1,t2;
  if(dev->shutdown)
    {
      return;
    }
  t1=MAX_INIT_SEND_TIME - time_difference(&dev->last_int_tv, NULL);
  t2=MAX_INIT_SEND_TIMEOUT - time_difference(&dev->send_tv, NULL);
  if(t1<0)t1=0;
  if(t2<0)t2=0;

  if(t2<t1)
    t1=t2; 
  mod_timer(&dev->initate_send_timer_list,t1?jiffies + usecs_to_jiffies(t1):jiffies+1);
}

static void inline add_detect_reset_timer(struct file* file)
{
  eib_t *dev = (eib_t*)file->private_data;
  long t1,t2,t3;
  if(dev->shutdown)
    {
      return;
    }
  t1=MIN_RESET_WAIT_TIME-time_difference(&dev->reset_tv, NULL);
  t2=READ_WRITE_TIMEOUT-time_difference(&dev->start_tv, NULL);
  t3=MAX_TIME_BETWEEN_INT-time_difference(&dev->last_int_tv, NULL);
  if(t1<0)t1=0;
  if(t2<0)t2=0;
  if(t3<0)t3=0;
  if(dev->state!=7)
    {
      t2=t1;
      if(t3<t1)
	t1=t3;
    }
  mod_timer(&dev->detect_reset_timer_list,t1?jiffies + usecs_to_jiffies(t1):jiffies+1);
}


/*********************************************************************
 ** Name    : bitcount
 ** Purpose : counts the 1s of a Byte 
 **
 **
 **
 ********************************************************************/
static int bitcount(unsigned char x) 
{
  int b;
 
  for(b = 0; x != 0; x >>= 1)
    if(x & 01) b++;
 
  return b;
}

/*********************************************************************
 ** Name    : time_difference
 ** Purpose : if second parameter == NULL => actual time will be used
 **
 **
 **
 ********************************************************************/
static inline long time_difference(struct timespec *tv1, struct timespec *tv2) 
{
  struct timespec foo;

  if(tv2 == NULL) {
    foo=CURRENT_TIME; 
    tv2 = &foo;
  }

  return     (tv2->tv_nsec - tv1->tv_nsec)/1000 +  /* usecs   */
    1000000 *(tv2->tv_sec - tv1->tv_sec);     /* seconds * 1000000 */
}



/*********************************************************************
 ** Name    : init_serial
 ** Purpose : No serial driver used/this module programs the UART
 **           itself (Speed: 9600Baud, 8 Databits, 1 Stoppbit)
 **           
 **
 ********************************************************************/
static void init_serial(eib_t *dev) 
{
  outb(0x00, dev->port + UART_IER);
  dev->old_lcr = inb(dev->port + UART_LCR);        /* save old line control reg */

  outb(UART_LCR_DLAB,  dev->port + UART_LCR); /* set divisor latch access bit in line control reg */
  dev->old_dll =        inb(dev->port + UART_DLL); /* save divisor latch low */
  dev->old_dlh =        inb(dev->port + UART_DLM); /* save divisor latch high 
					    * 115200 / Divisorlatch = speed
					    */
  outb(0x0c,           dev->port + UART_DLL); /* write 0x0c to divisor latch low */
  outb(0x00,           dev->port + UART_DLM); /* write 0x00 to divisor latch high 
					    * Speed : 9600
					    */
  
  outb(UART_LCR_WLEN8, dev->port + UART_LCR); /* 8 Databits in the line control reg */
  outb(0x00,           dev->port + UART_FCR); /* disable Fifo control reg */
  dev->old_mcr =        inb(dev->port + UART_MCR); /* save modem control reg */

  outb(UART_MCR_DTR | UART_MCR_OUT2, dev->port + UART_MCR);  
  /* Set RTS, Enable GP02 neccessary for UART ints */
  outb(UART_IER_MSI | UART_IER_RDI,  dev->port + UART_IER);  
  /* Enable modem status int and receiver data int */

  inb(dev->port + UART_RX);  /* read and trash old byte from buffer */
}

/*********************************************************************
 ** Name    : restore_serial
 ** Purpose : Restores the settings of the UART control regs from the
 **           variables explained above
 **
 **
 ********************************************************************/
static void restore_serial(eib_t *dev) {
  outb(0x00,          dev->port + UART_IER);
  outb(UART_LCR_DLAB, dev->port + UART_LCR); /* enable divisor latch access */
  outb(dev->old_dll,       dev->port + UART_DLL);
  outb(dev->old_dlh,       dev->port + UART_DLM);
  outb(dev->old_lcr,       dev->port + UART_LCR);
  outb(dev->old_mcr,       dev->port + UART_MCR);
}

/*********************************************************************
 ** Name    : eib_open
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static int eib_open(struct inode *inode, struct file *file) 
{
  eib_t *dev;     
  unsigned long flags;
  int port;
  int irq;
  int ports[4] = {0x3f8, 0x2f8, 0x3e8, 0x2e8}; 
  
  /* determine minor number (corresponds to serial port number) */
  unsigned int minor = iminor (inode);

  PDEBUG("open %d",minor);
  /* don't handle higher minors as number of devices supported */
  if (minor >= sizeof(used)/sizeof(used[0])) 
    return -ENODEV;  

  port=ports[minor];

  /* should we better probe for irqs? */
  if ((port == 0x3f8) || (port == 0x3e8))
    irq = 4;
  else
    irq = 3;

  /* we only let one process open the device */
  /* kernel code is never preempted if holding a spinlock */
  spin_lock(&open_lock);
  if (used[minor]) {
    spin_unlock(&open_lock);
    return -EBUSY;
  }
  used[minor]=1;
  spin_unlock(&open_lock);

  /* assign device-struct and initialize it */
  dev = (void*)kmalloc (sizeof (eib_t), GFP_KERNEL);
  if(!dev) goto outmem;
  memset (dev, 0, sizeof (eib_t));

  spin_lock_init(&dev->lock);
  spin_lock_init(&dev->irqlock);
  dev->shutdown=0;

  file->private_data = dev;
  dev->minor = minor;
  dev->port=port;
  dev->irq=irq;

  if(!request_region(dev->port, 0x08, "eib"))
    {
      kfree(dev);
      goto outdev;
    }
  
  if(request_irq(dev->irq, interrupt_handler, SA_INTERRUPT, "eib", (void*)file)<0)
    {
      release_region(dev->port, 0x08);
      kfree(dev);
      goto outdev;
    }
  
  PDEBUG("resouces allocated %d",dev->minor);
  spin_lock_irqsave(&dev->lock, flags);
  init_timer(&dev->initate_send_timer_list);
  init_timer(&dev->detect_reset_timer_list);
  dev->initate_send_timer_list.function = initate_send_timer;
  dev->initate_send_timer_list.data=(unsigned long int)(void*)file;
  dev->detect_reset_timer_list.function= detect_reset_timer;
  dev->detect_reset_timer_list.data=(unsigned long int)(void*)file;

  dev->state=0;
  dev->initate_send=0;
  dev->warning_flag=0;
  dev->reset_count=0;
  dev->last_int_tv=CURRENT_TIME;

  dev->read_head=0;
  dev->read_free=0;
  dev->read_next=0;
  dev->read_pos=0;
  spin_lock_init(&dev->read_lock);
  init_waitqueue_head(&dev->read_queue);

  dev->write_head=0;
  dev->write_free=0;
  dev->write_next=0;
  dev->write_pos=0;
  spin_lock_init(&dev->write_lock);
  init_waitqueue_head(&dev->write_queue);

  outb(0x00, dev->port+UART_IER);
  init_serial(dev);
  spin_unlock_irqrestore(&dev->lock, flags);
  tasklet_init(&dev->temt_tasklet, do_temt_bh,(unsigned long int)(void*)file);

  return 0;

 outmem:
	used[minor] = 0;
	return -ENOMEM;

 outdev:
	used[minor] = 0;
	return -ENODEV;
}

/*********************************************************************
 ** Name    : eib_release
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static int eib_release(struct inode *inode, struct file *file) 
{
  unsigned long flags;
  eib_t *dev = file->private_data;

  PDEBUG("close %d",dev->minor);
  dev->shutdown=1;

  flush_scheduled_work();
  synchronize_kernel();

  spin_lock_irqsave(&dev->lock, flags);
  restore_serial(dev);

  free_irq(dev->irq, (void*)file);
  release_region(dev->port, 0x08);

  del_timer_sync(&dev->initate_send_timer_list);
  del_timer_sync(&dev->detect_reset_timer_list);

  used[dev->minor] = 0;
  spin_unlock_irqrestore(&dev->lock, flags);

  PDEBUG("closed %d",dev->minor);
  /* must (obviously enough) be last */
  kfree (dev);

  return 0;
}


/*********************************************************************
 ** Name    : eib_cleanup_module
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
void __exit eib_cleanup_module(void) 
{


  unregister_chrdev(eib_major, "eib");

  PDEBUG("successfully unloaded!\n");
}

module_exit(eib_cleanup_module);

/* -------   */

// EXPORT_NO_SYMBOLS;

#define EIB_BCU_VERSION 1   /* eib bcu version */


MODULE_LICENSE("GPL v2");






#define setRTS() outb(UART_MCR_DTR | UART_MCR_RTS | UART_MCR_OUT2, dev->port + UART_MCR)
#define clrRTS() outb(UART_MCR_DTR | UART_MCR_OUT2, dev->port+UART_MCR)

#define getCTS() (inb(dev->port + UART_MSR) & UART_MSR_CTS)



/*
 * File Operation structure
 * 
 */
static struct file_operations eib_fops = 
 {
    .owner=      THIS_MODULE,
    .read=       eib_read,
    .write=      eib_write,
    .poll=       eib_poll,
    .ioctl=      eib_ioctl,
    .open=       eib_open,
    .flush=      eib_flush,
    .release=    eib_release,
  };





/* State 0 : CTS = 1 / RTS = 1
 *
 *
 *
 */



/*********************************************************************
 ** Name    : do_reset
 ** Purpose : Resets the serial port
 **
 **
 **
 ********************************************************************/
static void do_reset(eib_t* dev, unsigned long reason) 
{

#ifdef EIB_DEBUG
  printk(KERN_NOTICE "debug: do_reset: state=%d, reason=%lx, initate_send=%d, getCTS()=%d, MCR=%x\n", 
	 dev->state, 
	 reason, 
	 dev->initate_send, 
	 getCTS(), 
	 inb(dev->port + UART_MCR));
#endif

  dev->warning_flag |= reason;
  dev->state = 7;
  inb(dev->port + UART_IIR);
  inb(dev->port + UART_RX);
  inb(dev->port + UART_MSR);
  dev->reset_tv=CURRENT_TIME;
  clrRTS();
}



/*********************************************************************
 ** Name    : eib_read
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static ssize_t eib_read(struct file *file, 
			char __user *buf, 
			size_t count, 
			loff_t *offset) 
{
  eib_t *dev = (eib_t*)file->private_data;
  int rb_tail, len;
  unsigned long flags;

 retry:  
  while(dev->read_next==dev->read_free) 
    { 
      if(file->f_flags & O_NONBLOCK) /* read buffer empty */
	return -EAGAIN;
	wait_event_interruptible(dev->read_queue, !(dev->read_next==dev->read_free) );

      if(signal_pending(current))
	return -ERESTARTSYS;
    }
  
  spin_lock_irqsave(&dev->read_lock, flags);
  

  if(dev->read_next==dev->read_free) 
    {
      spin_unlock_irqrestore(&dev->read_lock, flags);
      goto retry;
    }

  rb_tail=dev->read_next;
  len = (dev->read_buffer[rb_tail][0] & 0x1f);

  if(count >= len)
      dev->read_next=(dev->read_next+1)%MAX_READ_BUFFER;

  spin_unlock_irqrestore(&dev->read_lock, flags);

  if(count < len)
    return -ENOMSG;

  if(copy_to_user(buf, &dev->read_buffer[rb_tail][1], len))
    return -EFAULT;

  spin_lock_irqsave(&dev->read_lock, flags);

  dev->read_head=(dev->read_head+1)%MAX_READ_BUFFER;

  spin_unlock_irqrestore(&dev->read_lock, flags);

  return len;
}


/*********************************************************************
 ** Name    : eib_write
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static ssize_t eib_write(struct file *file, 
			 const char __user *buf, 
			 size_t count, 
			 loff_t *offset) 
{
  eib_t *dev = (eib_t*)file->private_data;
  int temp, wb_head;
  unsigned long flags;

  if(count > 32) return -EINVAL;
 retry:
  while((dev->write_free+1)%MAX_WRITE_BUFFER==dev->write_head)
    {
      /* write buffer full */

    if(file->f_flags & O_NONBLOCK)
      return -EAGAIN;
    wait_event_interruptible(dev->write_queue,  !((dev->write_free+1)%MAX_WRITE_BUFFER==dev->write_head));
    
    if(signal_pending(current))
      return -ERESTARTSYS;
  }

  spin_lock_irqsave(&dev->write_lock, flags);

  if((dev->write_free+1)%MAX_WRITE_BUFFER==dev->write_head) {

    spin_unlock_irqrestore(&dev->write_lock, flags);
    goto retry;
  }


  wb_head=dev->write_free;
  dev->write_free=(dev->write_free+1)%MAX_WRITE_BUFFER;

  spin_unlock_irqrestore(&dev->write_lock, flags);

  temp = (count & 0x1f) | 0x20;
  dev->write_buffer[wb_head][0] = temp | (bitcount(temp) % 2) * 0x80;
  if(copy_from_user(&dev->write_buffer[wb_head][1], buf, count))
    return -EFAULT;
  dev->write_result[wb_head]=0;

  spin_lock_irqsave(&dev->write_lock, flags);

  dev->write_next=(dev->write_next+1)%MAX_WRITE_BUFFER;
  dev->send_tv=CURRENT_TIME;
  add_initate_send_timer(file);

  spin_unlock_irqrestore(&dev->write_lock, flags);
  if(file->f_flags & O_NONBLOCK)
    return count & 0x1f;

  while(!dev->write_result[wb_head])
    {
      /* write buffer full */

    wait_event_interruptible(dev->write_queue,  dev->write_result[wb_head]);
    
    if(signal_pending(current))
      return -EINTR;
  }

  if(dev->write_result[wb_head]>0)
    return count & 0x1f;
  else
    return 0;
}




/*********************************************************************
 ** Name    : eib_poll
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static unsigned int eib_poll(struct file *file, poll_table *wait) 
{
  eib_t *dev = (eib_t*)file->private_data;
  unsigned int mask = 0;
  unsigned long flags;

  poll_wait(file, &dev->read_queue, wait);
  poll_wait(file, &dev->write_queue, wait);

  spin_lock_irqsave(&dev->read_lock, flags);
  if(dev->read_free!=dev->read_next)  mask |= POLLIN | POLLRDNORM;
  spin_unlock_irqrestore(&dev->read_lock, flags);

  spin_lock_irqsave(&dev->write_lock, flags);
  if((dev->write_free+1)%MAX_WRITE_BUFFER!=dev->write_head)
    mask |= POLLOUT | POLLWRNORM;
  spin_unlock_irqrestore(&dev->write_lock, flags);

  return mask;
}




/*********************************************************************
 ** Name    : eib_ioctl
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static int eib_ioctl(struct inode *inode, 
		     struct file *file, 
		     unsigned int cmd, 
		     unsigned long arg)
{
  unsigned long flags;
  eib_t *dev = (eib_t*)file->private_data;

  switch(cmd) {
    
  case EIB_GET_BCU_VERSION:
    return EIB_BCU_VERSION;
    
  case EIB_GET_DEVICE_INFORMATION:
    break;

  case EIB_RESET_DEVICE_STATISTIC:
    spin_lock_irqsave(&dev->lock, flags);

    dev->warning_flag = 0;

    spin_unlock_irqrestore(&dev->lock, flags);

    break;

  case EIB_CLEAR_WRITE_BUFFER:
    spin_lock_irqsave(&dev->write_lock, flags);

    del_timer(&dev->initate_send_timer_list);
    add_detect_reset_timer(file);
    dev->initate_send=0;
    dev->warning_flag=0;
    dev->write_head=dev->write_free;
    do_reset(dev,0x00400000);

    spin_unlock_irqrestore(&dev->write_lock, flags);

    wake_up_interruptible_all(&dev->write_queue);
    break;

  case EIB_CLEAR_READ_BUFFER:
    spin_lock_irqsave(&dev->read_lock, flags);

    del_timer(&dev->initate_send_timer_list);
    add_detect_reset_timer(file);
    dev->warning_flag=0;
    dev->read_free=dev->read_next;
    do_reset(dev,0x00400000);

    spin_unlock_irqrestore(&dev->read_lock, flags);

    wake_up_interruptible_all(&dev->read_queue);
    break;

  case EIB_SINGLE_OPEN:
    break;

  default:
    return -ENOIOCTLCMD;
  }

  return 0;
}



/*********************************************************************
 ** Name    : eib_flush
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static int eib_flush(struct file *file) 
{  
  eib_t *dev = (eib_t*)file->private_data;
  while(dev->write_head!=dev->write_free) { // write buffer not empty
    wait_event_interruptible(dev->write_queue, !(dev->write_head!=dev->write_free));
    if(signal_pending(current))
      return -ERESTARTSYS;
  }

  return 0;
}






/*********************************************************************
 ** Name    : eib_init_module
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
int __init eib_init_module(void) 
{
  int retval;

  retval = register_chrdev(eib_major, "eib", &eib_fops);
  if(retval<0) {
    return retval;
  }

  if(eib_major == 0)  eib_major = retval;




  printk(KERN_INFO "eib driver loaded (%d)\n", eib_major);

  return 0;
}





/*********************************************************************
 ** Name    : initate_send_timer
 ** Purpose : ?
 **
 **
 **
 ********************************************************************/
static void initate_send_timer(unsigned long dev_id) 
{
  struct file *fp = (struct file*)dev_id;
  eib_t *dev = (eib_t*)fp->private_data;
  unsigned long flags;

  if(dev->write_head==dev->write_next) {
    EIB_PRINT("initate send with empty buffer");
    dev->warning_flag |= 0x10000; // initate send with empty write buffer
    return;
  }

  spin_lock_irqsave(&dev->lock, flags);

  if(dev->state==0 && !getCTS() && 
     time_difference(&dev->last_int_tv, NULL) > MAX_INIT_SEND_TIME) {

    dev->initate_send=1;
    dev->start_tv=CURRENT_TIME;
    dev->last_int_tv=CURRENT_TIME;

    setRTS();
    add_detect_reset_timer(fp);

    spin_unlock_irqrestore(&dev->lock, flags);

    return;
  }
  if(time_difference(&dev->send_tv, NULL) > MAX_INIT_SEND_TIMEOUT)
    {
      EIB_PRINT("timeout initiate send");

      do_reset(dev,0x10000000);
      add_detect_reset_timer(fp);

      spin_unlock_irqrestore(&dev->lock, flags);

      return;
    }

  add_initate_send_timer(fp);

  spin_unlock_irqrestore(&dev->lock, flags);
}




/*********************************************************************
 ** Name    : detect_reset_timer
 ** Purpose : ?
 **
 **
 **
 ********************************************************************/
static void detect_reset_timer(unsigned long dev_id) 
{
  struct file *fp = (struct file*)dev_id;
  eib_t *dev = (eib_t*)fp->private_data;
  unsigned long flags;

  spin_lock_irqsave(&dev->lock, flags);

  if(dev->state==0 && !dev->initate_send && !getCTS()) { // idle

    spin_unlock_irqrestore(&dev->lock, flags);

    return;
  } else if(dev->state==7) { // reset
    if(time_difference(&dev->reset_tv, NULL)>MIN_RESET_WAIT_TIME) {
      dev->last_int_tv=CURRENT_TIME;
      if(dev->write_head!=dev->write_next&&dev->reset_count>10)
	{
	  dev->reset_count=0;
	  dev->write_result[dev->write_head]=-1;
	  dev->write_head=(dev->write_head+1)%MAX_WRITE_BUFFER;
	  wake_up_interruptible_all(&dev->write_queue);
	}
      if(dev->write_head!=dev->write_next)
	{
	  dev->send_tv=CURRENT_TIME;
	  add_initate_send_timer(fp);
	}
      dev->read_pos=0;
      dev->write_pos=0;
      dev->state=0;
      dev->reset_count++;
      spin_unlock_irqrestore(&dev->lock, flags);

      return;
    }
  } else { // read || write
    if(time_difference(&dev->start_tv, NULL)>READ_WRITE_TIMEOUT)
      do_reset(dev,0x100000); // read write timeout
    else if(time_difference(&dev->last_int_tv, NULL)>MAX_TIME_BETWEEN_INT)
	do_reset(dev,0x200000); // max time between int exceeded
  }
  add_detect_reset_timer(fp);

  spin_unlock_irqrestore(&dev->lock, flags);
}




/*********************************************************************
 ** Name    : check_interrupt_pending
 ** Purpose : 
 **
 **
 **
 ********************************************************************/
static inline void check_interrupt_pending(eib_t* dev) 
{
  int id = inb(dev->port + UART_IIR);

  if(~id & UART_IIR_NO_INT) {    /* = 0x01 no int occured */
    if((id & UART_IIR_ID) == UART_IIR_RDI)
      do_reset(dev,0x40000); // unexpected receive data interrupt

    else 
      if(inb(dev->port + UART_MSR) & UART_MSR_DCTS)
	if(getCTS())
	  do_reset(dev,0x80000); // unexpected cts change to high
      // else !getCTS() -> state unchanged
    // else uninteresting modem status register change -> ignore
  }
}



/*********************************************************************
 ** Name    : do_temt_bh
 ** Purpose : Tasklet Bottom half interrupt handler
 **
 **
 **
 ********************************************************************/
static void do_temt_bh(unsigned long dev_id) 
{
  struct file *fp = (struct file*)dev_id;
  eib_t *dev = (eib_t*)fp->private_data;
  struct timespec tv;
  unsigned long flags;

  if(dev->state == 7) return;        /* reset */

  if(dev->state != 3 && dev->state != 6) {
    do_reset(dev,0x40);             /* unknown state */
    return;
  }

  tv=CURRENT_TIME;
  while((~inb(dev->port+UART_LSR) & UART_LSR_TEMT) && 
	time_difference(&tv, NULL)<MAX_TEMT_WAIT)
    ;                           /* busy waiting */

  if(~inb(dev->port+UART_LSR)&UART_LSR_TEMT)
    do_reset(dev,0x1000); // exceed MAX_TEMT_WAIT

  switch(dev->state) {

  case 3:
    if(dev->read_pos==(dev->read_buffer[dev->read_free][0]&0x1f)+1) { // read finished

      if(dev->read_buffer[dev->read_free][0]==0xa0)
	dev->warning_flag |= 0x20000; // read bcu reset
      else 
	{
	  dev->reset_count=0;
	  dev->read_free = (dev->read_free + 1) % MAX_READ_BUFFER;
	  wake_up_interruptible_all(&dev->read_queue);
	}

      dev->read_pos=0;

      spin_lock_irqsave(&dev->lock, flags);

      if(dev->write_head!=dev->write_next)
	{
	  dev->send_tv=CURRENT_TIME;
	  add_initate_send_timer(fp);
	}
      dev->state = 0;
    } 
    else 
      {
	spin_lock_irqsave(&dev->lock, flags);
	dev->state = 1;
      }
    check_interrupt_pending(dev);
    clrRTS();

    spin_unlock_irqrestore(&dev->lock, flags);

    break;

  case 6:
    tv=CURRENT_TIME;

    while(time_difference(&tv, NULL) < WRITE_TEMT_CLRRTS_DELAY);


    if(dev->write_pos==(dev->write_buffer[dev->write_head][0]&0x1f)+1) { // write finished

      dev->write_result[dev->write_head]=1;
      dev->reset_count=0;
      dev->write_head=(dev->write_head+1)%MAX_WRITE_BUFFER;
      wake_up_interruptible_all(&dev->write_queue);
      dev->write_pos=0;

      if(dev->write_head!=dev->write_next)
	{
	  dev->send_tv=CURRENT_TIME;
	  add_initate_send_timer(fp);
	}

      spin_lock_irqsave(&dev->lock, flags);


      dev->state = 0;

      check_interrupt_pending(dev);
      clrRTS();

    } /* if(wb_col... */
    else {

      check_interrupt_pending(dev);

      clrRTS();
      tv=CURRENT_TIME;
      while(time_difference(&tv, NULL) < WRITE_CLRRTS_SETRTS_DELAY);

      spin_lock_irqsave(&dev->lock, flags);

      dev->state=4;

      check_interrupt_pending(dev);
      setRTS();
    }

    spin_unlock_irqrestore(&dev->lock, flags);

    break;

  case 7: // reset
    break;

  default:
    do_reset(dev,0x40); // unknown state
    break;
  }
  dev->last_int_tv=CURRENT_TIME;
}






/*********************************************************************
 ** Name    : interrupt_handler
 ** Purpose : Performs the state machine
 **
 **
 **
 ********************************************************************/
static irqreturn_t interrupt_handler(int irq, 
		       void *dev_id, 
		       struct pt_regs *regs) 
{
  struct file *fp = (struct file*)dev_id;
  eib_t *dev = (eib_t*)fp->private_data;
  int tmp = 0, id;

  spin_lock(&dev->irqlock);
  id = inb(dev->port + UART_IIR);

  if(~id & UART_IIR_NO_INT) {              /* is there any interrupt ? */

    if(dev->state!=0 && dev->state!=7 && 
       time_difference(&dev->start_tv, NULL) > READ_WRITE_TIMEOUT)
      do_reset(dev,0x100000);                  /* read write timeout */

    switch(dev->state) {                        /* State machine state */ 
      
    case 0: // idle
      if((id & UART_IIR_ID) == UART_IIR_MSI){/* Modem status interrupt */
	if(inb(dev->port + UART_MSR) & UART_MSR_DCTS) {
	  if(getCTS()) {
	    if(dev->initate_send) {             /* write to bcu */
	      outb(dev->write_buffer[dev->write_head][0], dev->port + UART_TX);
	      dev->initate_send = 0;
	      dev->write_pos++;
	      dev->state = 5;
	      
	      break;
	    } 
	    else { // read
	      add_detect_reset_timer(fp);
	      if((dev->read_free+1)%MAX_READ_BUFFER==dev->read_head) {
		do_reset(dev,0x10);            /* read buffer overflow */
	      } 
	      else {
		dev->start_tv=CURRENT_TIME;
		setRTS();
		outb(0xFF, dev->port + UART_TX);
		dev->state = 2;
	      }
	    }
	  } // else !getCTS() -> state=0
	} // else uninteresting modem status register change -> ignore
      } 
      else do_reset(dev,0x2);                  /* unexpected receive data interrupt */
      
      break;
      
    case 1:
      if((id & UART_IIR_ID) == UART_IIR_MSI) {
	if(inb(dev->port + UART_MSR) & UART_MSR_DCTS)
	  if(getCTS()) {
	    setRTS();
	    outb(0x00, dev->port + UART_TX);
	    dev->state=2;
	  } // else !getCTS() -> state=1
	// else uninteresting modem status register change -> ignore
      } else
	do_reset(dev,0x200); // unexpected receive data interrupt
      break;
      
    case 2:
      if((id & UART_IIR_ID) == UART_IIR_RDI) {
	
	tmp = inb(dev->port + UART_RX);
	dev->read_buffer[dev->read_free][dev->read_pos++] = tmp;
	if(inb(dev->port + UART_LSR) & UART_LSR_OE) {
	  do_reset(dev,0x80); // overrun
	  break;
	}
	if(dev->read_pos==1 && !(((tmp & 0x60) == 0x20) && 
                          ((tmp & 0x80) == (bitcount(tmp & 0x7f) % 2) * 0x80))) {
	  do_reset(dev,0x400); // parity error
	  break;
	}
	dev->state=3;

	tasklet_schedule(&dev->temt_tasklet);   /* Instead of the Bottom half handler */
      }
      else if(inb(dev->port+UART_MSR)&UART_MSR_DCTS)
	if(getCTS())
	  do_reset(dev,0x20);                  /* unexpected cts change to high */
      // else !getCTS() -> ok
      // else uninteresting modem status register change
      break;
      
    case 3:

      if((id & UART_IIR_ID) == UART_IIR_RDI)
	do_reset(dev,0x4);                     /* unexpected receive data interrupt */
      
      else 
	if(inb(dev->port + UART_MSR) & UART_MSR_DCTS)
	  if(getCTS()) 
	    {
	      //dev->state = 1;
	      do_reset(dev,0x8);                 /* unexpected cts change to high */
	    }

      break;

    case 6:

      if((id & UART_IIR_ID) == UART_IIR_RDI)
	do_reset(dev,0x4);                     /* unexpected receive data interrupt */
      
      else 
	if(inb(dev->port + UART_MSR) & UART_MSR_DCTS)
	  if(getCTS())
	    do_reset(dev,0x8);                 /* unexpected cts change to high */
      // else !getCTS() -> state unchanged
      // else uninteresting modem status register change -> ignore
      break;

    case 4:

      if((id&UART_IIR_ID)==UART_IIR_MSI) {
	if(inb(dev->port+UART_MSR)&UART_MSR_DCTS) 
	  if(getCTS()) {
	    outb_p(dev->write_buffer[dev->write_head][dev->write_pos++], dev->port+UART_TX);
	    dev->state=5;
	  } // else !getCTS() -> state=4
	// else uninteresting modem status register change -> ignore
      } else
	do_reset(dev,0x800);                   /* unexpected receive data interrupt */
      break;
      
    case 5:
      if((id&UART_IIR_ID)==UART_IIR_RDI) {
	tmp = inb(dev->port+UART_RX);

	dev->state=6;

	if(inb(dev->port+UART_LSR)&UART_LSR_OE) {
	  do_reset(dev,0x80);                  /* overun */
	  break;
	}
	
	if(dev->write_pos==1 && tmp!=0xff) {
	    
	  dev->warning_flag|=0x100;             /* read forced */
	  dev->write_pos=0;
	  if((dev->read_free+1)%MAX_READ_BUFFER==dev->read_head) {
	    do_reset(dev,0x4000);              /* read forced && read buffer full */
	    
	    break;
	  }
	  
	  if(((tmp & 0x60) != 0x20) ||
	     ((tmp & 0x80) != (bitcount(tmp & 0x7f) % 2) * 0x80)) {
	    do_reset(dev,0x400);               /* parity error */
	    
	    break;
	  }
	  dev->read_buffer[dev->read_free][dev->read_pos++] = tmp;

	  dev->state=3;

	}
	if(dev->write_pos>1 && tmp!=0x00) {
	  do_reset(dev,0x2000);                /* unexpect byte received */
	  break;
	}

	tasklet_schedule(&dev->temt_tasklet);   /* Instead of the Bottom half handler */
      } 
      else 
	if(inb(dev->port+UART_MSR)&UART_MSR_DCTS)
	  do_reset(dev,0x8000);                /* unexpected cts change */
	// else uninteresting modem status register change
	break;
	
    case 7: // reset
      if((id&UART_IIR_ID)==UART_IIR_MSI)
	inb(dev->port + UART_MSR);
      else inb(dev->port + UART_RX);

      break;
      
      default:
	do_reset(dev,0x40); // unknown state	
    }
  } 
  else dev->warning_flag |= 0x1; // no interrupt pending


  dev->last_int_tv=CURRENT_TIME;
  spin_unlock(&dev->irqlock);
  return IRQ_HANDLED;
}
  


module_init(eib_init_module);   /* own function eib_init_module instead of
				 * "init_module"
				 */
