/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: bmrecv.c -- busmonitor receive function
 *
 *    $Id: bmrecv.c,v 1.2 2001/05/23 11:33:49 rst Exp $
 *
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    12-Sep-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/version.h>
#include "deque.h"
#include "tpuart_main.h"
#include "rxstm.h" /* for state defines */

/*
 * helper function: advance read pointer,
 * save data and wake up reading process
 */
static void save_data (tpuart_t *dev)
{
	
        dev->rx.read += 2;
	if (dev->rx.read >= (dev->rx.buffer + 2 * BUFSIZE))
		dev->rx.read = dev->rx.buffer;
	deque_put_irqsave (&dev->rx.temp, &dev->rx.tel_buffer);
	set_bit (NEW_TEL, &dev->rx.flags);
	if (dev->async_queue) {
		PDEBUG ("BM: Waking up process with SIGIO!\n");
		kill_fasync (&dev->async_queue, SIGIO, POLL_IN);
	       
	} else

	  wake_up_interruptible (&dev->rx.waitq);
}

/*
 * This is the worker routine in busmonitor mode.
 * Since we only have to propagate the data without
 * any processing, this is quite simple.
 */
void bm_recv (void *arg)
{
	tpuart_t *dev = arg;


	if (dev->rx.state == RX_CTRL) {
		dev->rx.count = 0;
		dev->rx.temp.timestamp = dev->rx.last_rx;
		memset (dev->rx.temp.data, 0, 64);
		/* if the received byte is not a valid control byte,
		   we just assemble a telegram with a single data byte */
		if ((*dev->rx.read & 0xd3) != L_DATA_CTRL &&
		    (*dev->rx.read & 0xd3) != L_LONG_DATA_CTRL) {
			dev->rx.temp.data[0] = *dev->rx.read;  
			dev->rx.temp.data_length = 1;
			save_data (dev);
			return;
		} else /* handle long telegrams here! */
			dev->rx.state = RX_DATA; /* we really only need something other
						    than RX_CTRL, so ... */
	} else if (dev->rx.count == 5) {
		dev->rx.rlength = *dev->rx.read & 0x0F;
		dev->rx.temp.data_length = dev->rx.rlength + 8;
	} else if ((dev->rx.count > 6) && (dev->rx.rlength-- <= 0)) {
		/* end of telegram */
		dev->rx.state = RX_CTRL;
		dev->rx.temp.data[dev->rx.count++] = *dev->rx.read;
		save_data (dev);
		return;
	}

	/* usual case */
	dev->rx.temp.data[dev->rx.count++] = *dev->rx.read;

	dev->rx.read += 2;
	if (dev->rx.read >= (dev->rx.buffer + 2 * BUFSIZE))
		dev->rx.read = dev->rx.buffer;

	return;
}
