/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: deque.c -- telegram buffer implementation
 *
 *    $Id: deque.c,v 1.2 2001/05/23 11:33:49 rst Exp $
 *
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    08-Sep-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include "tpuart_main.h"

/*
 * This function advances one of the head/tail-pointers
 * in deque and returns the new pointer
 */
static void *advance_ptr (deque_t *deque, void *p)
{
	PDEBUGG ("p is: 0x%x\n", p);
	if ((p + deque->elem_size) >= (deque->buffer + deque->bufsize)) {
		PDEBUGG ("new p is: 0x%x\n", deque->buffer);
		return (deque->buffer);
	}
	PDEBUGG ("new p is: 0x%x\n", p + deque->elem_size);
	return (p + deque->elem_size);
}  

/*
 * The macros below are mega-ugly. Skip them.
 * It is assumed that retval is defined before any macro call.
 * (Done in the __head macros)
 */
#define __deque_is_full(deque) do { \
	if (((deque)->head + (deque)->elem_size) >= ((deque)->buffer + (deque)->bufsize)) { \
	        if ((deque)->buffer == (deque)->tail) { \
			PDEBUGG ("deque->buffer = 0x%x, deque->tail = 0x%x; deque full!\n", (deque)->buffer, (deque)->tail); \
			retval = 1; \
		} else if (((deque)->buffer + (deque)->elem_size) == (deque)->tail) { \
			PDEBUGG ("deque->buffer = 0x%x, deque->tail = 0x%x; deque full!\n", (deque)->buffer, (deque)->tail); \
			retval = 1; \
		} else { \
			PDEBUGG ("deque->buffer = 0x%x, deque->tail = 0x%x; deque not full!\n", (deque)->buffer, (deque)->tail); \
			retval = 0; \
		} \
	} else if (((deque)->head + (deque)->elem_size) == (deque)->tail) { \
		PDEBUGG ("deque->head = 0x%x, deque->tail = 0x%x; deque full!\n", (deque)->head, (deque)->tail); \
		retval = 1; \
	} else { \
		PDEBUGG ("deque->head = 0x%x, deque->tail = 0x%x, " \
			"deque->buffer = 0x%x; deque not full!\n", (deque)->head, (deque)->tail, (deque)->buffer); \
	       retval = 0; \
	} \
} while (0)

#define __deque_get(elem, deque) do { \
	if ((deque)->tail == (deque)->head) \
		retval = 0; \
        else { \
	        memcpy ((elem), (deque)->tail, (deque)->elem_size); \
	        PDEBUGG ("copied data in deque_get ()!\n"); \
	        (deque)->tail = advance_ptr ((deque), (deque)->tail); \
	        retval = 1; \
       } \
} while (0)

#define __deque_put(elem, deque) do { \
	__deque_is_full (deque); \
	if (retval) \
                retval = 0; \
        else { \
	        memcpy ((deque)->head, (elem), (deque)->elem_size); \
	        PDEBUGG ("deque->head is: 0x%x\n", (deque)->head); \
	        PDEBUGG ("copied data in deque_put ()!\n"); \
	        (deque)->head = advance_ptr ((deque), (deque)->head); \
	        PDEBUGG ("deque->head is: 0x%x\n", (deque)->head); \
	        retval = 1; \
        } \
} while (0)

#define __deque_is_empty(deque) do { \
        if ((deque)->tail == (deque)->head) \
                retval = 1; \
        else \
	        retval = 0; \
} while (0)

#define __head(deque) \
        int retval = 0; \
        spin_lock (&(deque)->lock);

#define __tail(deque) do { \
        spin_unlock (&(deque)->lock); \
        return retval; \
} while (0)

#define __head_irq(deque) \
        int retval = 0; \
        spin_lock_irq (&(deque)->lock);

#define __tail_irq(deque) do { \
        spin_unlock_irq (&(deque)->lock); \
        return retval; \
} while (0)

#define __head_irqsave(deque) \
        int retval = 0; \
        unsigned long flags; \
        spin_lock_irqsave (&(deque)->lock, flags);

#define __tail_irqsave(deque) do { \
        spin_unlock_irqrestore (&(deque)->lock, flags); \
        return retval; \
} while (0)

/*
 * NOTE: All functions below return 0 on error or
 * 1 otherwise
 */

/*
 * This one is used for initializing the deque. For maximum
 * memory usage use powers of two for bufsize. elem_size
 * is the size of one element in the queue.
 */
int deque_init (deque_t *deque, size_t elem_size, int bufsize)
{
	deque->buffer = (void*)kmalloc (bufsize, GFP_KERNEL);
	if (deque->buffer == NULL) {
		return 0;
	}

	deque->head = deque->buffer;
	deque->tail = deque->buffer;

	deque->elem_size = elem_size;
	deque->bufsize = bufsize;

	spin_lock_init (&deque->lock);

	PDEBUGG ("created deque with buffer size: %d at: 0x%x\n", deque->bufsize, deque->buffer);
	PDEBUGG (".................... elem_size: %d (0x%x)\n", deque->elem_size, deque->elem_size);

	return 1;
}

/*
 * Free allocated memory and do cleanup (nothing to be done
 * for now)
 */
void deque_cleanup (deque_t *deque)
{
	kfree (deque->buffer);
}

/*
 * _irq functions simply clear and set the interrupt bit,
 * _irqsave use the save/restore_flags - macros,
 * the others use plain spinlocks
 */
int deque_get (void *elem, deque_t *deque)
{
	__head (deque);
	__deque_get (elem, deque);
	__tail (deque);
}

int deque_get_irq (void *elem, deque_t *deque)
{
	__head_irq (deque);
	__deque_get (elem, deque);
	__tail_irq (deque);
}

int deque_get_irqsave (void *elem, deque_t *deque)
{
	__head_irqsave (deque);
	__deque_get (elem, deque);
	__tail_irqsave (deque);
}

int deque_put (void *elem, deque_t *deque)
{
	__head (deque)
	__deque_put (elem, deque);
	__tail (deque);
}

int deque_put_irq (void *elem, deque_t *deque)
{
	__head_irq (deque);
	__deque_put (elem, deque);
	__tail_irq (deque);
}

int deque_put_irqsave (void *elem, deque_t *deque)
{
	__head_irqsave (deque);
	__deque_put (elem, deque);
	__tail_irqsave (deque);
}

int deque_is_empty (deque_t *deque)
{
	__head (deque);
	__deque_is_empty (deque);
	__tail (deque);
}

int deque_is_empty_irq (deque_t *deque)
{
	__head_irq (deque);
	__deque_is_empty (deque);
	__tail_irq (deque);
}

int deque_is_empty_irqsave (deque_t *deque)
{
	__head_irqsave (deque);
	__deque_is_empty (deque);
	__tail_irqsave (deque);
}

int deque_is_full (deque_t *deque)
{
	__head (deque);
	__deque_is_full (deque);
	__tail (deque);
}

int deque_is_full_irq (deque_t *deque)
{
	__head_irq (deque);
	__deque_is_full (deque);
	__tail_irq (deque);
}

int deque_is_full_irqsave (deque_t *deque)
{
	__head_irqsave (deque);
	__deque_is_full (deque);
	__tail_irqsave (deque);
}


