 /* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: heartbeat.c -- routines for the heartbeat
 *
 *    $Id: heartbeat.c,v 1.0 2004/10/2 08:23:13 rb Exp $
 *
 *    Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *    University of Technology Vienna
 *    
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    28-Sept-2004: 0.01  Started Coding
 *
 *    Last change:
 *
 */

#include <linux/sched.h>
#include <asm/io.h>

#include "tpuart_main.h"
#include "heartbeat.h"

extern int heartbeat_interval;

/*
 * init kernel timer and start heartbeat 
 */
void init_heartbeat(tpuart_t *dev) {
  
  set_bit(HEARTBEAT_ON, &dev->flags);
  
  /* init kernel timer */
  init_timer(&dev->heartbeat_timer);
  dev->heartbeat_timer.data=(unsigned long)dev;
  dev->heartbeat_timer.function=heartbeat;	     
  
  /* send first state request */
  set_bit(HB_STATE_AWAIT, &(dev)->flags);		     
  sending_state_message(dev);	
  
  /* start timer */
  dev->heartbeat_timer.expires=jiffies+HZ*heartbeat_interval;   
  add_timer(&dev->heartbeat_timer);   
  
  
}


/* 
 * delete kernel timer
 */
void del_heartbeat(tpuart_t *dev) {
  
  clear_bit(HEARTBEAT_ON, &(dev)->flags);   
  del_timer_sync(&dev->heartbeat_timer);


}


/*
 * send next state message or queue if 
 * a write is in progress.
 */
void sending_state_message(tpuart_t *dev) {

  unsigned long flags;

  spin_lock_irqsave(&dev->lock, flags);       //critical section (disable interrupts)
  
  if (test_bit(SEND_IN_PROGRESS, &dev->tx.flags)) {
    
    PDEBUG("Queue state message!\n");
    set_bit(STATE_MESS_QUEUED, &dev->flags);  //if we are sending a frame wait with state message
    
  } else {
    PDEBUG("Send state request!\n");
    outb(U_STATE_REQ, dev->port);          
  }
  
  spin_unlock_irqrestore(&dev->lock, flags);  // end critical section (enable interrupts)
  
}


/*
 * timer function for heartbeat (normal operation)
 *
 * controls arrival of last state message
 * and maybe starts reset loop.
 */
void heartbeat(unsigned long ptr)
{
  tpuart_t *dev=(tpuart_t *)ptr;

  /* don't do anything if heartbeat is off */
  if(!test_bit(HEARTBEAT_ON, &dev->flags)) {
    return;
  }

  PDEBUG("Heartbeat interrupt \n");  
 
  if(test_bit(HB_STATE_AWAIT, &dev->flags)) {    // no new state message received
   
    set_bit(DEAD, &dev->flags);
   
    /* wake up sleeping write or read */
    wake_up_interruptible(&dev->rx.waitq);
    wake_up_interruptible(&dev->tx.waitq);
    
    dev->heartbeat_timer.function=resetloop_request;        // reset tpuart 
    mod_timer(&dev->heartbeat_timer, jiffies);                //rerun immediately
    
    
  } else {   // state message received
    
    /* send next state message */
    set_bit(HB_STATE_AWAIT, &dev->flags); 
    sending_state_message(dev);
    
    /* restart timer */      
    mod_timer(&dev->heartbeat_timer, jiffies+HZ*heartbeat_interval);  //rerun in heartbeat_interval
        
      
  }
  
}

  
/*
 * timer function for heartbeat (failure)
 *
 * send reset request
 */ 
void resetloop_request(unsigned long ptr) {
  
  tpuart_t *dev=(tpuart_t *)ptr;
  
  /* don't do anything if heartbeat is off */
  if(!test_bit(HEARTBEAT_ON, &dev->flags)) {
    return;
  }

 
  /* sending a reset request */
  PDEBUG ("Resetting tpuart (heartbeat)\n");
  
  dev->resetted=0;
  outb (U_RESET_REQ, dev->port);      // reset TPUART 
                             

  dev->heartbeat_timer.function = resetloop_answer;
  mod_timer(&dev->heartbeat_timer, jiffies+HZ/16);       //rerun in ~60ms
  
  
}

/* 
 * timer function for heartbeat (failure)
 *
 * checks if reset was successful
 */
void resetloop_answer(unsigned long ptr) {
  
  tpuart_t *dev=(tpuart_t *)ptr;
  
  /* don't do anything if heartbeat is off */
  if(!test_bit(HEARTBEAT_ON, &dev->flags)) {
    return;
  }

 if (!dev->resetted) {
      TPUART_PRINT ("Reset failed!\n"); 
      dev->heartbeat_timer.function=resetloop_request;                // try it again
     
    }else {              //successful reset
    
      clear_bit(SEND_AND_WAIT, &dev->tx.flags);   //allow a new sending process
      
      clear_bit(HB_STATE_AWAIT, &dev->flags);  // send next state message
      sending_state_message(dev);
      dev->heartbeat_timer.function=heartbeat;   //continue with state messages    
      clear_bit(DEAD, &dev->flags);
    }

 dev->resetted = 0; 
  
 mod_timer(&dev->heartbeat_timer, jiffies+HZ*heartbeat_interval);    //rerun in heartbeat_interval
  
}
  
