/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: rxstm.c -- receive state machine
 *
 *    $Id: rxstm.c,v 1.4 2001/05/23 11:33:49 rst Exp $
 *
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    07-Aug-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/version.h>
//#include <asm/spinlock.h>
#include <asm/io.h>
#include "rxstm.h"
#include "deque.h"
#include "tpuart_main.h"

extern int is_my_addr (unsigned short, tpuart_t*);

typedef short stm_t;

static stm_t rx_ctrl (tpuart_t*);
static stm_t rx_source_high (tpuart_t*);
static stm_t rx_source_low (tpuart_t*);
static stm_t rx_dest_high (tpuart_t*);
static stm_t rx_dest_low (tpuart_t*);
static stm_t rx_npci (tpuart_t*);
static stm_t rx_tpci (tpuart_t*);
static stm_t rx_data (tpuart_t*);
static stm_t rx_chksum (tpuart_t*);
static stm_t rx_poll_chksum (tpuart_t*);
static stm_t rx_poll_data (tpuart_t*);


static void wake_process (tpuart_t *dev);

static stm_t (*action[NR_STATES]) (tpuart_t*) = {
	rx_ctrl,
	rx_source_high,
	rx_source_low,
	rx_dest_high,
	rx_dest_low,
	rx_npci,
	rx_tpci,
	rx_data,
	rx_chksum,
	rx_poll_chksum,
	rx_poll_data
};

/*
 * The main function of the state machine.
 * It reads all received bytes, updates the checksum
 * and calls the function for the relevant state.
 */
void rxstm (void *arg)
{
	tpuart_t *dev = arg;

	while (dev->rx.read != dev->rx.write) {
		dev->rx.checksum = ((*dev->rx.read & 0xff) ^ dev->rx.checksum);

	    
#ifdef TIMEOUT_DETECTION
	
		if (test_and_clear_bit (TIMEOUT, &dev->rx.flags)) {
		  dev->rx.state = action[RX_CHKSUM] (dev);
		}
		else {
		  dev->rx.state = action[dev->rx.state] (dev);
		}

#else

		dev->rx.state = action[dev->rx.state] (dev);


#endif

		++dev->rx.read;

		if (dev->rx.read >= (dev->rx.buffer + 2 * BUFSIZE))
		  dev->rx.read = dev->rx.buffer;
		

		}

}


/* NOTE: in the state functions dev->rx.read must be
   incremented _exactly_ once! */

/* Get the first byte and initialize the telegram buffer */
stm_t rx_ctrl (tpuart_t *dev)
{
	unsigned long flags = *(dev->rx.read + 1);
	PDEBUG ("State RX_CTRL called\n");

	if (!test_bit (CTRL, &flags)) {
		dev->rx.read++;
		return RX_CTRL;         /* wait for next ctrl byte */
	}

	/* fill out telegram buffer */
	dev->rx.temp.timestamp = dev->rx.last_rx;

	memset (dev->rx.temp.data, 0, 64);

	dev->rx.temp.data[RX_CTRL] = *dev->rx.read++;



	if (test_bit (POLLING, (unsigned long *) (dev->rx.read) ))
		set_bit (POLLING, &dev->rx.flags);
	return RX_SOURCE_HIGH;
}

stm_t rx_source_high (tpuart_t *dev)
{
	PDEBUG ("State RX_SOURCE_HIGH called\n");

	dev->rx.temp.data[RX_SOURCE_HIGH] = *dev->rx.read++;
	return RX_SOURCE_LOW;
}

 /* If we receive a telegram that we sent ourselves, we have to
   compare the whole received message with the sent one. The first
   indicators to such a case are our own physical address in the source
   address field of the message and the flag of the transmitter routine.
   We check this here. */
stm_t rx_source_low (tpuart_t *dev)
{
	unsigned short addr;
	PDEBUG ("State RX_SOURCE_LOW called\n");

	dev->rx.temp.data[RX_SOURCE_LOW] = *dev->rx.read++;
	addr = dev->rx.temp.data[RX_SOURCE_HIGH] << 8;
	addr |= dev->rx.temp.data[RX_SOURCE_LOW];
	PDEBUG ("received source address: 0x%x\n", addr);
	PDEBUG ("in our address space? %d\n", is_my_addr (addr, dev));

	if (is_my_addr (addr, dev) && 
	    test_and_clear_bit (DID_SEND, &dev->tx.flags)) {
		PDEBUG ("setting FROM_US... \n");
		set_bit (FROM_US, &dev->rx.flags);
	}
	return RX_DEST_HIGH;
}

stm_t rx_dest_high (tpuart_t *dev)
{
	PDEBUG ("State RX_DEST_HIGH called\n");

	dev->rx.temp.data[RX_DEST_HIGH] = *dev->rx.read++;
	return RX_DEST_LOW;
}

stm_t rx_dest_low (tpuart_t *dev)
{
	PDEBUG ("State RX_DEST_LOW called\n");

	dev->rx.temp.data[RX_DEST_LOW] = *dev->rx.read++;
	return RX_NPCI;
}

/* This is the state where we extract the telegram length.
   It counts from byte 7 (without checksum!) */
stm_t rx_npci (tpuart_t *dev)
{
	PDEBUG ("State RX_NPCI called\n");

       
	dev->rx.rlength = *dev->rx.read & 0x0F; /* extract lower nibble of NPCI */

	 	
	
	if (test_bit (POLLING, &dev->rx.flags)) {
		dev->rx.temp.data_length = dev->rx.rlength + 7;
		dev->rx.temp.data[RX_NPCI] = *dev->rx.read++;
		dev->rx.count = RX_NPCI + 1;
		return RX_POLL_CHKSUM;
	}
	

	dev->rx.temp.data_length = dev->rx.rlength + 8; /* includes chksum */
	dev->rx.temp.data[RX_NPCI] = *dev->rx.read++;

	/* copy flags from tel_buffer to flags variable */
	if (test_bit (FOR_US,(unsigned long *) (dev->rx.read) ))
		set_bit (FOR_US, &dev->rx.flags); 
	if (test_bit (RCV_BUSY, (unsigned long *) (dev->rx.read) ))
		set_bit (RCV_BUSY, &dev->rx.flags);
	
	return RX_TPCI;
}

stm_t rx_tpci (tpuart_t *dev)
{
	PDEBUG ("State RX_TPCI called\n");

	dev->rx.temp.data[RX_TPCI] = *dev->rx.read++;
	dev->rx.count = RX_TPCI + 1;    /* from now on we will use a position counter
					   for the array index into temp.data */

	if (dev->rx.rlength <= 0)
	        return RX_CHKSUM;

	return RX_DATA;
}

/* This state is called as long as there are data bytes. */
stm_t rx_data (tpuart_t *dev)
{
	PDEBUG ("State RX_DATA called\n");



	dev->rx.temp.data[dev->rx.count++] = *dev->rx.read++;

#ifndef TIMEOUT_DETECTION

	if (--dev->rx.rlength <= 0)
	  return RX_CHKSUM;
	
#endif
	


	return RX_DATA;
}

static inline void mark_for_us (volatile long *flags)
{
	/* since we already tested the source addresses on equality
	   above (lower source address state) and we obviously didn't send 
	   the just received message, we know that another device has 
	   the same source address as we have. */
	TPUART_PRINT ("WARNING: Detected duplicated source address on"
		      " the bus!\n");
	set_bit (FOR_US, flags);
}

/* Here we test the checksum. If it's ok, we queue the telegram,
   otherwise we throw it away (for now) */
stm_t rx_chksum (tpuart_t *dev)
{
	PDEBUG ("State RX_CHKSUM called\n");


#ifdef TIMEOUT_DETECTION

	*dev->rx.read++;                 //consume the dummy
#else

	dev->rx.temp.data[dev->rx.count] = *dev->rx.read++;

#endif


	if (dev->rx.checksum != 0x00) {
		PDEBUG ("Checksum error!\n");
		set_bit (CHKSUMERR, &dev->rx.flags);
	} else {
	       

		if (test_and_clear_bit (FROM_US, &dev->rx.flags)) {
			/* repeated telegrams are treated the same as non repeated ones.
			   I. e. we don't care about the repeat-flag in the control byte. 

                           check if the telegram is really the one sent by us, if not mark it
                           for us and handle it in next if statement */

			clear_bit (FOR_US, &dev->rx.flags);
			if ((dev->tx.temp.data[0] & 0xdf) != (dev->rx.temp.data[0] & 0xdf)) {
			  PDEBUG("1");
			  mark_for_us (&dev->rx.flags);
			}
			if (dev->tx.temp.data_length != dev->rx.temp.data_length) {
				 PDEBUG("2");
			  mark_for_us (&dev->rx.flags);
			}
			if (memcmp (dev->tx.temp.data + 1, dev->rx.temp.data + 1,
				    dev->rx.temp.data_length -2) != 0) {
				 PDEBUG("3");
			  mark_for_us (&dev->rx.flags);
			}
		}
 		if (test_bit (FOR_US, &dev->rx.flags)) {
			/* only queue tel if we didn't send busy before ('cause it
			   will come again) */
			if (!test_bit (RCV_BUSY, &dev->rx.flags)) 
				deque_put_irq (&dev->rx.temp, &dev->rx.tel_buffer);
		}
	}

	wake_process (dev);
  
	return RX_CTRL; /* await a new message */
}

/* the same in green, this time for polling mode */
stm_t rx_poll_chksum (tpuart_t *dev)
{
	PDEBUG ("State RX_POLL_CHKSUM called\n");
	dev->rx.temp.data[dev->rx.count++] = *dev->rx.read++;
	if (dev->rx.checksum != 0x00) {
		PDEBUG ("Checksum error!\n");
		set_bit (CHKSUMERR, &dev->rx.flags);
		clear_bit (POLLING, &dev->rx.flags);
		return RX_CTRL;
	} else {
		unsigned short saddr, daddr;
		saddr = dev->rx.temp.data[RX_SOURCE_HIGH] << 8;
		saddr |= dev->rx.temp.data[RX_SOURCE_LOW];
		daddr = dev->rx.temp.data[RX_DEST_HIGH] << 8;
		daddr |= dev->rx.temp.data[RX_DEST_LOW];

		/* we are only interested in polling frames we sent ourselves */
		if (!is_my_addr (saddr,dev)) 
			clear_bit (POLLING, &dev->rx.flags);
	}

	/* if more polling data follows, switch to next state. */
	if (dev->rx.rlength)
		return RX_POLL_DATA;

	deque_put_irq (&dev->rx.temp, &dev->rx.tel_buffer);
	wake_process (dev);

	return RX_CTRL;
}

stm_t rx_poll_data (tpuart_t *dev)
{
	PDEBUG ("State RX_POLL_DATA called\n");
	if (!test_bit (POLLING, &dev->rx.flags)) { /* just consume bytes if we didn't poll */
		if (--dev->rx.rlength)
			return RX_POLL_DATA;
		else {
			set_bit (NEW_TEL, &dev->rx.flags);
			clear_bit (POLLING, &dev->rx.flags);
			return RX_CTRL;
		}
	}
	
	dev->rx.temp.data[dev->rx.count++] = *dev->rx.read++;
	
	if (--dev->rx.rlength) 
		return RX_POLL_DATA;

	deque_put_irq (&dev->rx.temp, &dev->rx.tel_buffer);
	wake_process (dev);

	return RX_CTRL;
}

/* Wakes up waiting processes. When in polling mode, this should be called
   only if we are polling master! */
void wake_process (tpuart_t *dev)
{
	if (!deque_is_empty_irq (&dev->rx.tel_buffer)) {
		if (dev->async_queue) {
			PDEBUG ("Waking up process with SIGIO!\n");
		kill_fasync (&dev->async_queue, SIGIO, POLL_IN);

		} else if (test_bit (POLLING, &dev->rx.flags)) {
			PDEBUG ("Waking up process in tx waitqueue!\n");
			set_bit(WAKE_UP_WRITE, &dev->tx.flags);
			wake_up_interruptible (&dev->tx.waitq);
		} else {
			PDEBUG ("Waking up process in rx waitqueue!\n");
			wake_up_interruptible (&dev->rx.waitq);
		}
	}

	dev->rx.checksum = 0xFF;
	clear_bit (FOR_US, &dev->rx.flags);
	clear_bit (RCV_BUSY, &dev->rx.flags);
	clear_bit (POLLING, &dev->rx.flags);
}

/* That's all folks! */
