  /* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: tpuart_main.c -- main file
 *
 *    $Id: tpuart_main.c,v 1.6 2001/05/28 08:23:13 rst Exp $
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *    
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *    
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    31-Jul-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/28 08:23:13 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/major.h> 
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/slab.h>
#include <linux/moduleparam.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/system.h>

#include "tpuart_main.h"
#include "rxstm.h"
#include "utility.h"
#include "heartbeat.h"


/*
 * Module parameter definition
 */

int tpuart_major;
int heartbeat_interval=DEFAULT_HB_INTERVAL;

module_param(tpuart_major, int , 0);
MODULE_PARM_DESC (tpuart_major, "Major number for device (default=dynamic)");

module_param(heartbeat_interval, int, DEFAULT_HB_INTERVAL);
MODULE_PARM_DESC (heartbeat_interval, "Interval length between checks of the TPUART device (default=30 sec)");

MODULE_AUTHOR ("Raffael Stocker");
MODULE_DESCRIPTION ("Device-Driver for TP-UART EIB-Connector, V"
		     TPUART_VERSION ); 

MODULE_LICENSE("GPL v2");

/* private minor number use count */
static int used[4];

/* prototypes for file-operations */
static ssize_t tpuart_read (struct file *, char *,
			    size_t, loff_t *);

static ssize_t tpuart_write (struct file *, const char *, 
			     size_t, loff_t *); 

static int tpuart_open (struct inode *,
			struct file *);

static int tpuart_release (struct inode *,
			   struct file *);

static int tpuart_ioctl (struct inode *,
			 struct file *,
			 unsigned int, unsigned long);

static int tpuart_fasync (int,
			  struct file *,
			  int);

static unsigned int tpuart_poll (struct file *,
			poll_table *);


static struct file_operations tpuart_fops = {
	.owner =    THIS_MODULE,
	.read =     tpuart_read,
	.write =    tpuart_write,
	.poll =     tpuart_poll,
	.ioctl =    tpuart_ioctl,
	.open =     tpuart_open,
	.release =  tpuart_release,
	.fasync =   tpuart_fasync
};


extern int is_timeout (struct timeval, struct timeval);
extern int is_addressed (tpuart_t*);
extern int is_my_group (unsigned short, tpuart_t*);
extern int is_my_addr (unsigned short, tpuart_t*);
extern int ser_init (tpuart_t*);
extern void rxstm (void*);
extern void bm_recv (void*);
extern void transmit (void*);

static int __init tpuart_init (void)
{
 	int major;

	PDEBUG("tpuart major :%i\n", tpuart_major);
	/* set default major (defined in tpuart.h) if not supplied */
	if (tpuart_major == 0)
		tpuart_major = TPUART_MAJOR;

	/* register module at given major */
	if ((major = register_chrdev (tpuart_major, TPUART_NAME, &tpuart_fops)) < 0) {
		TPUART_PRINT ("unable to get major %d\n", tpuart_major);
		return major;
	}

	/* if we allocated a major number dynamically, remember the real major we got */
	if (tpuart_major == 0)
		tpuart_major = major; 

	TPUART_PRINT ("TP-UART-Driver, Version "TPUART_VERSION " (C) 2000, 2001 Raffael Stocker, "
		      "University of applied sciences, Deggendorf. " 
                      "Ported to kernel 2.6 in 2004 by Reinhold Buchinger, University of Technology Vienna "
		      "Covered by GNU GPL V2\n");
       	
	PDEBUG("Hearbeat interval:%i\n", heartbeat_interval);
	return 0;

}

static void __exit tpuart_cleanup (void)
{
	unregister_chrdev (tpuart_major, TPUART_NAME);

	return;
}


module_init (tpuart_init);
module_exit (tpuart_cleanup);




#define NEW_TEL_INIT \
        do { \
          do_gettimeofday (&dev->rx.last_rx); \
          clear_bit (NEW_TEL, &dev->rx.flags); \
          set_bit (CTRL, &tel_flags); \
          dev->rx.first = dev->rx.write; \
          *dev->rx.write++ = rcv_byte; \
          *dev->rx.write++ = tel_flags; \
          dev->rx.received = 1; \
          QUEUE_BH (&dev->rx.worker); \
        } while (0)
        
#define TX_WAKE_UP \
        do { \
	  set_bit (WAKE_UP, &flags); \
	  clear_bit (SEND_AND_WAIT, &dev->tx.flags); \
	  if (test_bit (SEND_REQU, &dev->tx.flags)) \
	          QUEUE_BH (&dev->tx.worker); \
        } while (0)


/* The central working routine. Data is collected here. This is our
   interface to the universe. */
static irqreturn_t int_handler (int irq, void *filp, struct pt_regs *regs)
{
	struct file *fp = (struct file*)filp;
	tpuart_t *dev = (tpuart_t*)fp->private_data;
	struct timeval tmp_time;

	unsigned long linestatus = inb (dev->port+LSR);
	unsigned long intident = inb (dev->port+IIR);
	unsigned long tel_flags = 0;
	u8 rcv_byte = 0;
	unsigned long flags = 0;

	while (!test_bit (INTP, &intident)) {
		switch (intident & INF)	{
		case RLSI:
			/* I never got such an error while testing.
			   We just ignore them... */
			PDEBUG ("receiver line status interrupt\n");
			if (test_bit (OERR, &linestatus)) {
				PDEBUG ("Overrun error\n"); 
			}
			if (test_bit (PERR, &linestatus)) { 
				PDEBUG ("Parity error\n"); 
			}
			if (test_bit (FERR, &linestatus)) {
				PDEBUG ("Framing error\n"); 
			}
			if (test_bit (BINT, &linestatus)) { 
				PDEBUG ("Break detect\n"); 
			}
			break;
		case RHI:  /* receiving ... */
			PDEBUG ("receiver interrupt\n");
			
			/* first read all bytes we get in a big fat while loop... */
			while (inb (dev->port+LSR) & DRDY) {
				/* NOTE: dev->rx.write must be set only twice
				   per iteration! */
				rcv_byte = inb (dev->port);
				
				/* handle state messages */
				if ((rcv_byte & TP_STATE_MASK) == TP_STATE_MASK) {
				  PDEBUG("Got state message\n");
				  dev->tpuart_state  = (unsigned char) rcv_byte;
				  clear_bit(HB_STATE_AWAIT, &dev->flags);                 //indicate heartbeat that state message has arrived
				
				  if (test_and_clear_bit(STATE_AWAIT, &dev->flags)) {     //ioctl state request is waiting for answer
				    wake_up_interruptible(&dev->tx.waitq);
				  }
				}

				/* NOTE: timing may not be accurate! */
				if (test_bit (BUSMON, &dev->rx.flags)) {
					*dev->rx.write++ = rcv_byte;
					*dev->rx.write++ = 0; /* keep flags zero in bm mode */
					do_gettimeofday (&dev->rx.last_rx);
					QUEUE_BH (&dev->rx.worker);
				} else if (test_bit (NEW_TEL, &dev->rx.flags)) {
		        new_tel:	switch (rcv_byte) {
			                case 0:
						/* skip all zero bytes */
						break;
			                case RESET_IND:  /* we were resetted */
						dev->resetted = 1;
						clear_bit (BUSMON, &dev->rx.flags); /* just in case we are in bm */
						INIT_WORK(&dev->rx.worker, rxstm, dev);
						break;
               		                case L_POLL_DATA_CTRL:
						PDEBUG ("got polling request\n");
						/* queue data to send */
						outb (U_POLLING_STATE | (dev->polling_slot & 0xf), dev->port);
						outb ((dev->polling_addr >> 8) & 0xff, dev->port);
						outb (dev->polling_addr & 0xff, dev->port);
						outb (dev->polling_resp, dev->port);
						set_bit (POLLING, &tel_flags);
						set_bit (INT_POLLING, &dev->rx.flags); /* this is for use
											  in the inthandler
											  (see below) */
						NEW_TEL_INIT;
						break;
		                        case L_DATA_NO_CONF:
						PDEBUG ("Got NO_CONF!\n");
						set_bit(NO_CONF, &dev->tx.flags);
						TX_WAKE_UP;
						break;
			                case TP_RECV_ERROR:
			                
						/* since we already may have two new bytes in
						   the fifo, we may get 2 tp_recv_errors */
						if (!test_bit (SEND_IN_PROGRESS, &dev->tx.flags)) {
							PDEBUG ("Got parity/checksum error!\n");
							set_bit (SENDERR, &dev->tx.flags);
						} else {
							PDEBUG ("Got first parity/checksum error!\n");
							clear_bit (SEND_IN_PROGRESS, &dev->tx.flags);
							set_bit (RESET_FIRST, &dev->tx.flags);
							break;
						}
						/* fall-thru */
				        case L_DATA_CONF:
						PDEBUG("Got Data_Conf!\n");
					        TX_WAKE_UP;
						break;
			                case TP_TRANSM_ERROR:
					        PDEBUG("Got Transmission Error!\n");
					        set_bit (SENDERR, &dev->tx.flags);
						set_bit (RESET_FIRST, &dev->tx.flags);
						TX_WAKE_UP;
						break;
			                default: 
						if ((rcv_byte & 0xd3) == L_DATA_CTRL) { 
							PDEBUG ("starting new telegram: CTRL= 0x%x\n", rcv_byte);
							NEW_TEL_INIT;
						}
			               }
				} else {
			
				 
				        tmp_time = dev->rx.last_rx;
					do_gettimeofday (&dev->rx.last_rx);
			
					
					if (is_timeout (tmp_time, dev->rx.last_rx)) {
					   
                                                PDEBUG ("TIMEOUT at 0x%x\n", rcv_byte);

#ifdef TIMEOUT_DETECTION
						
						set_bit (TIMEOUT, &dev->rx.flags);
						set_bit (NEW_TEL, &dev->rx.flags);
						*dev->rx.write++ = 0;            /* insert dummy element */
						*dev->rx.write++ = 0;            /* insert dummy tel_flags */
						QUEUE_BH (&dev->rx.worker);
						goto new_tel;                /* detected timeout (but received already next byte), start new ctrl-byte search */
						
#else
						
						outb (U_RESET_REQ, dev->port);      /* reset TPUART */
					        dev->rx.state = RX_CTRL;            /* reset receive state machine */
						set_bit (NEW_TEL, &dev->rx.flags);
						
						return IRQ_HANDLED;  /* leave ISR, all remaining bytes are useless if we reset TPUART */
						
#endif
					}
		  
					*dev->rx.write++ = rcv_byte;
					
		   
					if (++dev->rx.received == 6) {
						/* this part tests the address and sends
						   the Acknowledge */
						if (!test_bit (INT_POLLING, &dev->rx.flags) && is_addressed (dev)) {
							if (deque_is_full (&dev->rx.tel_buffer)) {
								PDEBUG ("Buffer is full, sending busy!\n");
								outb (U_ACK_INFO_BUSY, dev->port);
								set_bit (RCV_BUSY, &tel_flags);
							} else {
								outb (U_ACK_INFO_ACK, dev->port);
								set_bit (FOR_US, &tel_flags);
							}
						}
						dev->length = rcv_byte & 0x0F; 
					}
					*dev->rx.write++ = tel_flags;
					
#ifndef TIMEOUT_DETECTION 

					if ((dev->rx.received > 7) && (dev->length-- == 0))
					  set_bit (NEW_TEL, &dev->rx.flags); /* message ended (presumably),
										      await a new */
#endif

					QUEUE_BH (&dev->rx.worker);
				}
				PDEBUG ("rcv_byte: 0x%x\n", rcv_byte);

				if (dev->rx.write >= (dev->rx.buffer + 2 * BUFSIZE)) /* wrap around */
					dev->rx.write = dev->rx.buffer;
			}
			if (test_and_clear_bit (WAKE_UP, &flags)) {
				if (!(fp->f_flags & O_NONBLOCK)) {
					PDEBUG ("Wake up!\n");  
					set_bit(WAKE_UP_WRITE, &dev->tx.flags);
					wake_up_interruptible (&dev->tx.waitq);
				}
			}
			break;
		case THI:  /* sending ... */
		{
			short i = 0;
			PDEBUG ("transmitter interrupt\n");
			
			/* a state message is waiting to be send */
			if(((dev->tx.read)-1 == dev->tx.end) && (test_and_clear_bit(STATE_MESS_QUEUED, &dev->flags))) {    
			  outb(U_STATE_REQ, dev->port);
			  break;
			}
			
			
			if (!test_bit (SEND_IN_PROGRESS, &dev->tx.flags)) {
			  break;
			}

			/* test, if there's data to send */
			while ((dev->tx.read <= dev->tx.end) && (i++ < 2)) {
				PDEBUG ("tx.read: 0x%x\n", *dev->tx.read);
				outb (*dev->tx.read++, dev->port);
			}
			if ( (dev->tx.read)-1 == dev->tx.end) {              /* things we do if we are done */
			 
			  clear_bit (SEND_IN_PROGRESS, &dev->tx.flags);
				
			 
			}
		       break;
		}
		case TIP:
			PDEBUG ("time out interrupt (gee, how's that?)\n");
			break;
		case MSI:
			PDEBUG ("modem status interrupt (can't happen!)\n");
			break;
		default:
			PDEBUG ("unknown interrupt (can't happen!)\n");
		}
		/* loop while there are interrupts pending */
		intident = inb (dev->port+IIR);
	}

	return IRQ_HANDLED;
}

int tpuart_open (struct inode *inode, struct file *filp) 
{
	tpuart_t *dev;     
	spinlock_t open_lock;
	int ports[4] = {0x3f8, 0x2f8, 0x3e8, 0x2e8}; 
	
	/* determine minor number (corresponds to serial port number) */
	unsigned int minor = iminor (inode);

	PDEBUG ("minor: %d\n", minor);

	spin_lock_init(&open_lock);

	/* don't handle higher minors as number of devices supported */
	if (minor >= NR_DEVICES) 
		return -ENODEV;  

	/* we only let one process open the device */
	/* kernel code is never preempted if holding a spinlock */
	spin_lock(&open_lock);
	if (used[minor] == 1) {
	  spin_unlock(&open_lock);
	  return -EBUSY;
	}

	/* we may sleep below this point */
	used[minor] = 1;
	spin_unlock(&open_lock);

	/* FIXME: This really shouldn't be here... */
	/*if (check_region (ports[minor], 8) != 0) {
	  TPUART_PRINT ("IO 0x%x is in use\n", ports[minor]);
	  return -ENODEV;
		} else*/

	request_region (ports[minor], 8, TPUART_NAME); 

	/* assign device-struct and initialize it */
	dev = (void*)kmalloc (sizeof (tpuart_t), GFP_KERNEL);
	memset (dev, 0, sizeof (tpuart_t));
	filp->private_data = dev;
	dev->port = ports[minor];
	dev->minor = minor;

	/* should we better probe for irqs? */
	if ((dev->port == 0x3f8) || (dev->port == 0x3e8))
		dev->irq = 4;
	else
		dev->irq = 3;


	/* the buffers for the interrupt handler */
	dev->tx.buffer = (void*)kmalloc (BUFSIZE, GFP_KERNEL);
	dev->rx.buffer = (void*)kmalloc (2*BUFSIZE, GFP_KERNEL); /* half of the
								    space is for
								    flags */

	init_waitqueue_head (&dev->rx.waitq);
	init_waitqueue_head (&dev->tx.waitq);


	/* initialize spinlock */
	spin_lock_init(&dev->lock);


	/* already allocated memory is not freed! */
	if ((dev->tx.buffer == 0) || (dev->rx.buffer == 0)) {
		goto outmem;
	}

	/* huge space for group addresses; this is hard-coded, 'cause
	   we would get bad indexing errors in the group-address helpers
	*/
	dev->group_addr = (void*)vmalloc (8192);
	if (dev->group_addr == 0) {
		goto outmem;
	}

	dev->phys_addr = (void*)vmalloc (8192);
	if (dev->phys_addr == 0) {
		goto outmem;
	}

	memset (dev->group_addr, 0, 8192);
	memset (dev->phys_addr, 0, 8192);
	INSERT_GROUP_ADDR (0, dev); /* 0x0000 is broadcast address */

	dev->tx.write = dev->tx.read = dev->tx.end = dev->tx.buffer;
	dev->rx.write = dev->rx.read = dev->rx.first = dev->rx.buffer;

	if (!deque_init (&dev->tx.tel_buffer, sizeof (telegram_t), TXQ_SIZE)) {
		goto outmem;
	}

	if (!deque_init (&dev->rx.tel_buffer, sizeof (telegram_t), RXQ_SIZE)) {
		goto outmem;
	}

	/* since we are at the beginning of processing, we await a new telegram */
	set_bit (NEW_TEL, &dev->rx.flags);
	dev->rx.checksum = 0xff;

	/* set up bottom halves */
	INIT_WORK(&dev->rx.worker, rxstm, dev);
	INIT_WORK(&dev->tx.worker, transmit, dev);
	

	/* serial port initialization follows here */
	if (!ser_init (dev))
		goto outdev;
  
	/* now finally register interrupt handler */
	if (request_irq (dev->irq, int_handler, SA_INTERRUPT|SA_SHIRQ, "tpuart", (void*)filp) != 0) {
		TPUART_PRINT ("Couldn't get irq %d\n", dev->irq);
		goto outdev;
	}  

	PDEBUGG ("registered with port 0x%x and irq %d\n", dev->port, dev->irq);


	
	/* now that we have configured the serial port, we can initialize the TPUART
	   this has to be the first byte to send to the TPUART */
	dev->resetted=0;
	outb (U_RESET_REQ, dev->port);      /* reset TPUART */
	sleep_on_timeout (&dev->tx.waitq, HZ/16); /* wait 50 - 60 ms */

	/* This is our only check wether the tpuart is present.
	   Be careful if you have a device on the port that
	   triggers an explosion upon reception of the TPUART-Reset
	   byte...
	*/
	if (!dev->resetted) {
		TPUART_PRINT ("Communication with TPUART failed!\n");
		free_irq (dev->irq, filp);
		goto outdev;
	}
	dev->resetted = 0;

	/* heartbeat timer */
      	PDEBUG("start heartbeat!\n");
	init_heartbeat(dev);
	
	
	return 0;

 outmem:
	used[minor] = 0;
	return -ENOMEM;
 outdev:
	used[minor] = 0;
	return -ENODEV;
}

int tpuart_release (struct inode *inode, struct file *filp) 
{
	tpuart_t *dev = filp->private_data;
	unsigned long flags;


	/* turn off hearbeat */
	 del_heartbeat(dev);

	tpuart_fasync (-1, filp, 0);

	spin_lock_irqsave(&dev->lock, flags);

	outb (0x00, dev->port+MCR); /* switch off power supply for optocouplers */
	free_irq (dev->irq, filp);
	

	release_region (dev->port, 8);
      
        /* free memory */
	kfree ((void*)dev->rx.buffer);
	kfree ((void*)dev->tx.buffer);
	
	deque_cleanup (&dev->rx.tel_buffer);
	deque_cleanup (&dev->tx.tel_buffer);

	vfree ((void*)dev->group_addr);
	vfree ((void*)dev->phys_addr);
  
	/* mark this minor as free */
	used[dev->minor] = 0;

	spin_unlock_irqrestore(&dev->lock, flags);

	/* must (obviously enough) be last */
	kfree (dev);
  

	return 0;
}



/* We simply write the data (if any) from our tel_buffer to
   the userland buffer. If no data is present, sleep, unless
   O_NONBLOCK is set. */
ssize_t tpuart_read (struct file *filp,
		     char *buffer, size_t count, loff_t *ppos) 
{
         tpuart_t *dev = filp->private_data;
         telegram_t temp;
  
	if (count < sizeof (telegram_t))
		return -EINVAL;

	
	if (test_bit(DEAD, &dev->flags)) {
	  PDEBUG("no write possible - EREMOTEIO\n");
	  return -EREMOTEIO;   // no heartbeat anymore  
	}
	
	    
	/* non-blocking case */
	if (deque_is_empty_irq (&dev->rx.tel_buffer) && (filp->f_flags & O_NONBLOCK)) {   //return if no message is available
	  PDEBUG ("EAGAIN!\n");
	  return -EAGAIN;
	}

	PDEBUG ("Sleeping in read()!\n");
	
	
	/* sleep */
	if(wait_event_interruptible(dev->rx.waitq, (!deque_is_empty_irq(&dev->rx.tel_buffer)) || (test_bit(DEAD, &dev->flags)))) {
	  
	  return -ERESTARTSYS; //signal
	}
	

	if (test_bit(DEAD, &dev->flags)){
	  PDEBUG("EREMOTEIO in read\n");
	  return -EREMOTEIO;   //  no heartbeat anymore    

	} else {
	  deque_get_irq (&temp, &dev->rx.tel_buffer);       //get last received telegram
	  
	  return copy_to_user (buffer, &temp, sizeof (telegram_t)) ? -EFAULT : sizeof (telegram_t);
	}
}  
	

/* The same the other way around: We take the data from userland
   and store it in the tel_buffer. This again may sleep. */
ssize_t tpuart_write (struct file *filp,
		      const char *buffer, size_t count, loff_t *ppos)
{
	tpuart_t *dev = filp->private_data;
	ssize_t retval = 0;
	telegram_t temp;

	DEFINE_WAIT(wait);
	
	if (test_bit(DEAD, &dev->flags)) {
	  PDEBUG("no write possible - EREMOTEIO\n");
	  return -EREMOTEIO;   // no heartbeat anymore  
	}
	

	if (test_bit (BUSMON, &dev->rx.flags))         //no write in busmonitor mode
		return -EPERM;

	if (count < sizeof (telegram_t))
		return -EINVAL;
  
	while (deque_is_full_irq (&dev->tx.tel_buffer)) {
	
	  PDEBUG ("TX-Queue is full!\n");
		
	  if (filp->f_flags & O_NONBLOCK)    //return immediately in non-blocking case 
	    return -EAGAIN;
	  if (signal_pending (current))
	    return -ERESTARTSYS;
	  
	  PDEBUG ("Sleeping in write()!\n");
	  
	  prepare_to_wait(&dev->tx.waitq, &wait, TASK_INTERRUPTIBLE);
	 
	  if(deque_is_empty_irq (&dev->tx.tel_buffer))
	    schedule();
	 
	  finish_wait(&dev->tx.waitq, &wait);	//sleep if deque is full   
	}

	retval = copy_from_user (&temp, buffer, sizeof (telegram_t));

	if (retval != 0)
		return retval;
	
	deque_put_irq (&temp, &dev->tx.tel_buffer);

	PDEBUG ("Starting sending subroutine (sss)\n");
	QUEUE_BH (&dev->tx.worker);                      //queue bottom half

	/*clear all write status bits*/
	clear_bit(CONF, &dev->tx.flags);
	clear_bit(NO_CONF, &dev->tx.flags);
	clear_bit(SENDERR, &dev->tx.flags);
	clear_bit(DIED_BEFORE_CONF, &dev->tx.flags);

	
	if (filp->f_flags & O_NONBLOCK)                //don't wait for transmission and confirm in non-blocking case
		return sizeof (telegram_t);

	/* sleep */
	if (wait_event_interruptible(dev->tx.waitq, test_and_clear_bit(WAKE_UP_WRITE,&dev->tx.flags) || test_bit(DEAD, &dev->flags))) { 
	  return -ERESTARTSYS; /* signal */
	}
	  
	 
	PDEBUG ("Awakening from sleep!\n");
	
	/* reset if an error has occured */
	if (test_and_clear_bit (RESET_FIRST, &dev->tx.flags)) {
	  stop_heartbeat(dev);
	  PDEBUG ("Resetting tpuart...\n");
	  dev->resetted=0;
	  outb (U_RESET_REQ, dev->port);      /* reset TPUART */
	  sleep_on_timeout (&dev->tx.waitq, HZ/16); /* wait 50 - 60 ms */
	  
	  if (!dev->resetted) {
	    TPUART_PRINT ("Reset failed!\n");
	  }
	  dev->resetted = 0;
	  start_heartbeat(dev);
	}
	
       
	if (test_bit(DEAD, &dev->flags)) {
	  PDEBUG("EREMOTEIO in write");
	  return -EREMOTEIO;   /* no heartbeat anymore  */
	 



	}  else if (test_bit (NO_CONF, &dev->tx.flags)) {
	  return -ECOMM;                          /* last repetition is transmitted and no ACK received */
	  
	} else if (test_bit (SENDERR, &dev->tx.flags)) {
	  return -EIO;                             /* sender error */

	} else {
	  return sizeof (telegram_t);
	}
	    
}


int tpuart_fasync (int fd, struct file *filp, int mode)
{
	tpuart_t *dev = filp->private_data;

	return fasync_helper (fd, filp, mode, &dev->async_queue);
}

unsigned int tpuart_poll (struct file *filp, poll_table *wait)
{
	tpuart_t *dev = filp->private_data;
	unsigned int mask = 0;

	poll_wait (filp, &dev->rx.waitq, wait);
	poll_wait (filp, &dev->tx.waitq, wait);
	if (!deque_is_empty_irq (&dev->rx.tel_buffer))
		mask |= POLLIN | POLLRDNORM;
	if (!deque_is_full_irq (&dev->tx.tel_buffer))
		mask |= POLLOUT | POLLWRNORM;

	return mask;
}
	
int tpuart_ioctl (struct inode *inode, struct file *filp,
		  unsigned int command, unsigned long arg) 
{
	tpuart_t *dev = filp->private_data;
	unsigned short tmp_addr;
	unsigned long flags;
	int retval;
  
	/* check validity of commands */
	if (_IOC_TYPE (command) != TPUART_MAGIC) { 
	  return -EINVAL;
	}
	
	if (_IOC_NR (command) > TPUART_IOCMAX) {
	  return -EINVAL;
	}
	
	/* here we do the actual work */
	switch (command) {
	case TPUART_SET_PH_ADDR:
		retval = get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		INSERT_PHYS_ADDR (tmp_addr, dev);
		break;
	case TPUART_UNSET_PH_ADDR:
		retval =  get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		CLEAR_PHYS_ADDR (tmp_addr, dev);
		break;
	case TPUART_HAVE_PH_ADDR:
		retval =  get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		if (is_my_addr (tmp_addr, dev))
			return put_user (1, (unsigned short*)arg) ? -EFAULT : 0;
		else
			return put_user (0, (unsigned short*)arg) ? -EFAULT : 0;
		break;
	case TPUART_SET_GR_ADDR:
		retval =  get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		INSERT_GROUP_ADDR (tmp_addr, dev);
		break;
	case TPUART_UNSET_GR_ADDR:
		retval =  get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		if (tmp_addr != 0) /* don't allow deletion of group address 0 (broadcast) */
			CLEAR_GROUP_ADDR (tmp_addr, dev);
		break;
	case TPUART_HAVE_GR_ADDR:
		retval = get_user (tmp_addr, (unsigned short*)arg);
		if (retval)
			return -EFAULT;
		if (is_my_group (tmp_addr, dev))
			return put_user (1, (unsigned short*)arg) ? -EFAULT : 0;
		else
			return put_user (0, (unsigned short*)arg) ? -EFAULT : 0;
		break;
	case TPUART_SET_GR_ADDR_ARRAY:
	{
		unsigned short addresses[16];
		int i;
		
		copy_from_user ((unsigned short*)addresses, (unsigned short*)arg,
				sizeof (addresses));
	 
		for (i = 0; i < 16; i++) {
			if (addresses[i] == 0)
				break;
			INSERT_GROUP_ADDR (addresses[i], dev);
		}
	       
		break;
	}
	case TPUART_UNSET_GR_ADDR_ARRAY:
	{
		unsigned short addresses[16];
		int i;

		copy_from_user ((unsigned short*)addresses, (unsigned short*)arg,
				sizeof (addresses));
		
		for (i = 0; i < 16; i++) {
			if (addresses[i] == 0)
				break;
			CLEAR_GROUP_ADDR (addresses[i], dev);
		}
	       
		break;
	}
	case TPUART_SET_PH_ADDR_ARRAY:
	{
		unsigned short addresses[16];
		int i;
 
		copy_from_user ((unsigned short*)addresses, (unsigned short*)arg,
				sizeof (addresses));
	       
		for (i = 0; i < 16; i++) {
			if (addresses[i] == 0)
				break;
			INSERT_PHYS_ADDR (addresses[i], dev);
		}
		
		break;
	}
	case TPUART_UNSET_PH_ADDR_ARRAY:
	{
		unsigned short addresses[16];
		int i;

		copy_from_user ((unsigned short*)addresses, (unsigned short*)arg,
				sizeof (addresses));
	       
		for (i = 0; i < 16; i++) {
			if (addresses[i] == 0)
				break;
			CLEAR_PHYS_ADDR (addresses[i], dev);
		}
	       
		break;
	}
	case TPUART_BUSMON_OFF: 
	case TPUART_RESET:
	  
	       stop_heartbeat(dev);
	       		   
	       clear_bit (BUSMON, &dev->rx.flags); /* just in case we are in bm */
	       INIT_WORK(&dev->rx.worker, rxstm, dev);
		
	       dev->resetted=0;
	       outb (U_RESET_REQ, dev->port);      /* reset tp-uart */
	       sleep_on_timeout (&dev->tx.waitq, HZ/16);
	       if (!dev->resetted /*test_and_clear_bit (RESET_OK, &dev->rx.flags)*/){
		 start_heartbeat(dev);	
		 return -EIO;
		 
	       }
	       clear_bit(DEAD, &dev-> flags);   // we are alive
	       dev->resetted=0;
	       
	       PDEBUG("RESET\n");

	       start_heartbeat(dev);
	       
	       break;
	case TPUART_BUSMON_ON:
	       if(test_bit(DEAD, &dev->flags)){
		 return -EREMOTEIO;              //don't allow to enter busmonitor mode if we are dead
 	         }

	        stop_heartbeat(dev);    //no heartbeat in busmonitor mode
	        
		               
		set_bit (BUSMON, &dev->rx.flags);
		dev->rx.state = RX_CTRL;
	
		INIT_WORK(&dev->rx.worker, bm_recv, dev);     //use bm_recv as bottom half in busmonitor mode
	      
		outb (U_ACTIVATE_BUSMON_REQ, dev->port);
		break;
	case TPUART_SET_POLLING_SLOT:
		return get_user (dev->polling_slot, (unsigned char*)arg) ? -EFAULT : 0;
		break;
	case TPUART_GET_POLLING_SLOT:
		return put_user (dev->polling_slot, (unsigned char*)arg) ? -EFAULT : 0;
		break;
	case TPUART_SET_POLLING_ADDR:
		return get_user (dev->polling_addr, (unsigned short*)arg) ? -EFAULT : 0;
		break;
	case TPUART_GET_POLLING_ADDR:
		return put_user (dev->polling_addr, (unsigned short*)arg) ? -EFAULT : 0;
		break;
	case TPUART_SET_POLLING_RESP:
		return get_user (dev->polling_resp, (unsigned char*)arg) ? -EFAULT : 0;
		break;
	case TPUART_GET_POLLING_RESP:
		return put_user (dev->polling_resp, (unsigned char*)arg) ? -EFAULT : 0;
		break;
	
	/* get TPUART state */
	case TPUART_GET_STATE:
	  {
	        int tmp;
		
		if(test_bit(DEAD, &dev->flags)) {
		  PDEBUG("EREMOTEIO in TPUART_GET_STATE\n");
		  return -EREMOTEIO;
		}

		set_bit(STATE_AWAIT, &dev->flags);
		sending_state_message(dev);

		/* wait for State response message */
		if (wait_event_interruptible (dev->tx.waitq, (!test_bit(STATE_AWAIT,&dev->flags)) || test_bit(DEAD, &dev->flags)) ) {
		  return -ERESTARTSYS; /* signal */
		}
		
		if (test_bit(DEAD, &dev->flags)){
		  PDEBUG("EREMOTEIO in TPUART_GET_STATE\n");
		  return -EREMOTEIO;   /* no heartbeat anymore  */
		  
		} else {
		  spin_lock_irqsave(&dev->lock, flags);     //avoid inconsistent tpuart_state
		  tmp= dev->tpuart_state;
		  spin_unlock_irqrestore(&dev->lock, flags);
		  return put_user (tmp, (unsigned char*)arg) ? -EFAULT : 0;
		  break;
		}

	  }
        /* get last TPUART write status */
	case TPUART_GET_WRITE_STATUS:
	  
	  if(test_bit(CONF, &dev->tx.flags)){
	    return CONF;
	  } else if(test_bit(NO_CONF, &dev->tx.flags)){
	    return NO_CONF;
	  } else if(test_bit(SENDERR, &dev->tx.flags)){
	    return SENDERR;
	  } else if(test_bit(DIED_BEFORE_CONF, &dev->tx.flags)){
	    return DIED_BEFORE_CONF;
	  } else {
	    return STATE_AWAITING;
	  }
	  break;
	  
	default:
		PDEBUG ("fell in default...\n");
		return -EINVAL;
	}
	
	return 0;
}
