/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: tpuart_main.h -- general definitions
 *
 *    $Id: tpuart_main.h,v 1.2 2001/05/23 11:33:49 rst Exp $ 
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    31-Jul-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#ifndef __TPUART_MAIN
#define __TPUART_MAIN
 
#include <linux/ioctl.h>


#ifdef __KERNEL__

#include <linux/time.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/version.h>
#include <linux/workqueue.h>
//#include <asm/spinlock.h>
#include "deque.h"

/* module-defines */

#define TPUART_MAJOR    120     /* set default major here (0 = dynamic) */

#define TPUART_NAME     "tpuart"

#define NR_DEVICES      4     /* number of supported devices (4 serial ports) */

#ifndef MODULE_VERSION_TPUART
# warning "VERSION not set -- assuming 0.01"
# define TPUART_VERSION "0.01"
#else
# define TPUART_VERSION MODULE_VERSION_TPUART
#endif

#undef TPUART_PRINT
#define TPUART_PRINT(fmt, args...) printk(KERN_NOTICE "tpuart: " fmt, ## args)

#undef PDEBUG
#ifdef TPUART_DEBUG
#  define PDEBUG(fmt, args...) printk(KERN_NOTICE "tpuart: " fmt, ## args)
#else
# define PDEBUG(fmt, args...) ;
#endif

#undef PDEBUGG
#define PDEBUGG(fmt, args...)

#define BUFSIZE               256      /* size of transmit and receive ring-buffer */
#define RXQ_SIZE              2048     /* size of receive queue */
#define TXQ_SIZE              2048     /* size of transmit queue */
#define TEL_TIMEOUT           5000L    /* timeout between two telegrams */

#define DEFAULT_HB_INTERVAL    30      /* default heartbeat interval in sec */

/*
 * different modes for detection of frame end. 
 * evaluation of length information is default,
 * define TIMEOUT_DETECTION for evaluation of timeout
 */
//#define  TIMEOUT_DETECTION
 



/*
 * This is the structure that holds the telegrams
 */
typedef struct {
	struct timeval timestamp;          /* timestamp for received telegrams */
	unsigned short data_length;        /* length of data (below)
					      Note: this is _not_ the same as
					      NPCI[0:3]!!! */
	u8 data[64];                       /* telegram data */
} telegram_t;

typedef struct {
	u8 *buffer;                /* receive ring-buffer */
	u8 *write;                 /* write pointer in buffer */
	u8 *read;                  /* read pinter in buffer */
	u8 *end;                   /* points to byte with error condition
				      (e. g. timeout) */
	u8 *first;                 /* points to first byte in tel queue */
	unsigned short received;   /* counter for received bytes */
	struct timeval last_rx;    /* time of last received byte */
	deque_t tel_buffer;
	telegram_t temp;           /* used for filling out a message */
	char rlength;              /* holds restlength of telegram */
	short count;               /* writing position in data array (rx)
				      send count (tx);
				   */
	short state;               /* states for rxstm */
	unsigned char checksum;    /* used by rxstm for checkum calculation */
	struct work_struct worker;
	volatile unsigned long flags;


	wait_queue_head_t waitq;

} xmit_t;

/* This structure holds information relevant to the tp-uart */
typedef struct {
	int port;                    /* serial port number */
	unsigned int irq;            /* irq number */
	int minor;                   /* minor number */
	struct fasync_struct *async_queue;
/*	unsigned short phys_addr;*/    /* physical address */
	unsigned char *group_addr;   /* group address space */
	unsigned char *phys_addr;    /* physical address space */
	unsigned char polling_slot;  /* polling slot to use */
	unsigned short polling_addr; /* polling address */
	unsigned char polling_resp;  /* response to send upon polling request */
	unsigned char tpuart_state;  /* internal state of tp-uart */
        volatile unsigned long flags; /* general flags needed */
        char length;                 /* kludgy length count for inthandler */
	volatile long resetted;
	/* these structs are used by the interrupt handler and the state machines */
	xmit_t rx;
	xmit_t tx;
       /* spinlock for mutual exclusive access to serial port */
        spinlock_t lock;
       /* heartbeat */ 
       struct timer_list heartbeat_timer;
       
} tpuart_t;

/* Flag definitions for tpuart_t 
 * use only for internal services or administrative things
 */
/* read flags -- must be greater than 7 (see SHARED below) or 0, 1*/
#define NEW_TEL                8    /* indicates new telegram */ 
#define CHKSUMERR              9    /* indicate checksum error */
#define DID_SEND               10   /* indicate a sent telegram */
#define BUSMON                 11   /* set if we are in busmonitor mode */
#define FROM_US                12   /* this telegram is from us */
//#define LONG_DATA              0    /* processing long telegram */
#define INT_POLLING            1    /* indicate polling mode -- ONLY
				       FOR INTERRUPT HANDLER */


/* write flags */
#define CONF                   0    /* indicates receive of L_DATA_CONF_frame */
#define NO_CONF                1    /* indicates receive of L_DATA_NO_CONF frame */
#define SENDERR                2    /* couldn't send telegram */
#define SEND_AND_WAIT         3 /* int_handler is sending or already waiting for answer*/
#define SEND_IN_PROGRESS       4    /* indicate data waiting */
#define SEND_REQU              5    /* indicate send request */
#define RESET_FIRST            6    /* indicate a tpuart error condition */
#define WAKE_UP_WRITE          7    /* indicate wake up condition in write */
#define DIED_BEFORE_CONF       8   /* heartbeat died before conf was received */

/* additional return value for ioctl TPUART_GET_WRITE_STATUS (beside CONF, NO_CONF, SENDERR, DIED_BEFORE_CONF) */
#define STATE_AWAITING         3    /* no state response until now */



/* Flag definitions for int_handler
 */
#define WAKE_UP                0    /* wake up sleeping process */

/* Flag definitions for rx buffer
   SHARED flags can be set in both rx.flags _and_ in the rx buffer flag field
   bit no. can be max. 7
 */
#define CTRL                   0    /* indicates control byte */
#define TIMEOUT                1    /* indicates telegram timeout */
#define FOR_US                 3    /* SHARED this telegram is for us */
#define RCV_BUSY               4    /* SHARED set if we are busy */
#define POLLING                5    /* SHARED set if we are in polling mode */


/* general flags */
#define HEARTBEAT_ON          0     /* flag to stop heartbeat timer */
#define HB_STATE_AWAIT        1      /* waiting for new state message by heartbeat  - cleared by ISR */
#define STATE_AWAIT           2      /* indicates ioctl that new state message was received - cleared by ISR */
#define DEAD                  3      /* tpuart seems to be dead */
#define STATE_MESS_QUEUED     4     /* heartbeat couldn't send state message immediately */

/* 
 * Service code definitions for services TO tp-uart.
 * Names are (mostly) taken from the tp-uart specification
 */

#define U_RESET_REQ            0x01    /* reset tp-uart */
#define U_STATE_REQ            0x02    /* request internal communication state */
#define U_ACTIVATE_BUSMON_REQ  0x05    /* activate busmonitor-mode */
#define U_ACK_INFO_ACK         0x11    /* acknowledge information */
#define U_ACK_INFO_NACK        0x15    /* nack */
#define U_ACK_INFO_BUSY        0x13    /* busy */
#define U_L_DATA_START         0x80    /* start data transfer to tp-uart */
#define U_POLLING_STATE        0xE0    /* transmit polling information, slot number
					  must be added */


/*
 * Service code definitions for services FROM tp-uart.
 */

/* priority class must be extracted from these! */
#define L_DATA_CTRL            0x90    /* control byte: normal data */
#define L_LONG_DATA_CTRL       0x10    /* control byte: long data */
#define L_POLL_DATA_CTRL       0xF0    /* control byte: polling frame */

#define RESET_IND              0x03    /* indicates reset */
#define L_DATA_CONF            0x8B    /* indicates successful transmission */
#define L_DATA_NO_CONF         0x0B    /* indicates unsuccessful transmission */

/* TPUART Error Conditions */
#define TP_RECV_ERROR          0x47    /* checksum, parity or bit error */
#define TP_TRANSM_ERROR        0x27


#define TP_STATE_MASK          0x07   /* all state indication messages end with 0x07 */


/*
 * Register defininitions for serial port
 */

/* add these to your port number */
#define DLLB                   0       /* Devisor Latch Low Byte (DLAB == 1) */
#define IER                    1       /* Interrupt Enable Register */
#define DLHB                   1       /* Devisor Latch High Byte (DLAB == 1) */
#define IIR                    2       /* Interrupt Identification Register */
#define FCR                    2       /* FIFO Control Register */
#define LCR                    3       /* Line Control Register */
#define MCR                    4       /* Modem Control Register */
#define LSR                    5       /* Line Status Register */
#define MSR                    6       /* Modem Status Register */
#define SCR                    7       /* Scratch Register */
/* bit definitions for serial port registers */
/* IER (read-write) */
#define RIE                    0x01    /* Enable Received Data Available Interrupt */
#define TIE                    0x02    /* Enable Transmitter Holding Register Empty
					  Interrupt */
#define LSIE                   0x04    /* Enable Receiver Line Status Interrupt */
/* IIR (read-only) */
#define INTP                   0       /* Interrupt Pending Flag (1 = no int) */
#define INF                    0x06    /* INterrupt information Flags */
#define RLSI                   0x06    /* Receiver Line Status Interrupt */
#define THI                    0x02    /* Transmitter Interrupt */
#define RHI                    0x04    /* Receiver Interrupt */
#define MSI                    0x00    /* Modem Status Interrupt */
#define TIP                    0x08    /* Time-out Interrupt Pending */
#define FIND                   0xC0    /* Fifo INDication flag: (Bits 6:7)
					  00: no FIFO
					  01: FIFO enabled but unusable
					  11: FIFO enabled */
/* FCR (write-only) */
#define FEN                    0x01    /* Fifo ENable */

/* LCR (read-write) */
#define WLEN                   0x03    /* Word LENgth: (Bits 0:1)
					  00: 5 Bits
					  01: 6 Bits
					  10: 7 Bits
					  11: 8 Bits (we use this) */
#define PSEL                   0x18    /* Parity SELect: (Bits 3:5)
					  xx0: no parity
					  001: odd parity
					  011: even parity (used here)
					  101: high parity (sticky)
					  111: low parity (sticky) */
#define DLAB                   0x80    /* Divisor Latch Access Bit */
/* MCR (read-write) */
#define FDTR                   0x01    /* Force Data Terminal Ready */
#define FRTS                   0x02    /* Force Request To Send */
#define OUT2                   0x08    /* aux OUTput 2 */
/* LSR (read-only) */
#define DRDY                   0x01    /* Data ReaDY MASK*/
#define OERR                   1    /* Overrun ERRor */
#define PERR                   2    /* Parity ERRor */
#define FERR                   3    /* Framing ERRor */
#define BINT                   4    /* Break INTerrupt */
#define THE                    5    /* Transmitter Holding register Empty */
#define DHE                    6    /* Data Holding register Empty */
#define FRERR                  7    /* ERRor in Received Fifo */
/* MSR (read-only) */
#define DCTS                   0x01    /* Delta Clear To Send */
#define DDSR                   0x02    /* Delta Data Set Ready */
#define TERI                   0x04    /* Trailing Edge Ring Indicator */
#define DDCD                   0x08    /* Delta Data Carrier Detect */
#define CTS                    0x10    /* Clear To Send */
#define DSR                    0x20    /* Data Set Ready */
#define RIND                   0x40    /* Ring INDicator */
#define CD                     0x80    /* Carrier Detect */

#endif /* __KERNEL__ */


/* 
 * EIB-Telegram field definitions
 */

#define CTRL_BYTE              0       /* zeroth byte is control field */
#define SOURCE_ADDR_HIGH       1       /* MSD of source address */
#define SOURCE_ADDR_LOW        2       /* LSD of source address */
#define DEST_ADDR_HIGH         3       /* MSD of destination address */
#define DEST_ADDR_LOW          4       /* LSD of destination address */
#define NPCI                   5       /* holds information about addressing type,
					  routing count and telegram length 
				          flag definitions below: (logical and with
					  NPCI to extract information) */
# define GROUP_ADDR            0x80    /* if set group addressing, else physical
					  addressing */
# define ROUT_COUNT            0x70    /* lower three bits in upper nibble: hold
					  routing count */
# define TEL_LENGTH            0x0F    /* lower nibble: holds telegram length */
#define TPCI                   6       /* FIXME: Dunno what this is */
#define APCI                   7       /* FIXME: ditto */

/* TPUART IOCTL's */

#define TPUART_MAGIC           'E'     /* for construction of ioctl's */
#define TPUART_IOCMAX          22      /* maximum number of ioctl's */

/* set a physical address of host */
#define TPUART_SET_PH_ADDR  _IOW  (TPUART_MAGIC, 2, unsigned short)
/* check, if a physical address is set */
#define TPUART_HAVE_PH_ADDR  _IOWR  (TPUART_MAGIC, 3, unsigned short)
/* clear a physical address */
#define TPUART_UNSET_PH_ADDR _IOW (TPUART_MAGIC, 18, unsigned short)
/* set up to 16 physical addresses (excluding 0x0000) */
#define TPUART_SET_PH_ADDR_ARRAY _IOW (TPUART_MAGIC, 19, unsigned short[16])
/* clear 16 physical addresses */
#define TPUART_UNSET_PH_ADDR_ARRAY _IOW (TPUART_MAGIC, 20, unsigned short[16])
/* set a group address */
#define TPUART_SET_GR_ADDR  _IOW  (TPUART_MAGIC, 4, unsigned short)
/* clear a group address */
#define TPUART_UNSET_GR_ADDR _IOW (TPUART_MAGIC, 5, unsigned short)
/* check, if a group address is set */
#define TPUART_HAVE_GR_ADDR _IOWR (TPUART_MAGIC, 6, unsigned short)  
/* set 16 group addresses */
#define TPUART_SET_GR_ADDR_ARRAY _IOW (TPUART_MAGIC, 7, unsigned short[16])
/* clear 16 group addresses */
#define TPUART_UNSET_GR_ADDR_ARRAY _IOW (TPUART_MAGIC, 17, unsigned short[16])
/* send reset request to TPUART */
#define TPUART_RESET         _IO (TPUART_MAGIC, 8)
/* turn busmonitor mode ON */
#define TPUART_BUSMON_ON     _IO (TPUART_MAGIC, 9)
/* turn busmonitor mode OFF */
#define TPUART_BUSMON_OFF    _IO (TPUART_MAGIC, 10)
/* set polling slot */
#define TPUART_SET_POLLING_SLOT _IOW (TPUART_MAGIC, 11, unsigned char)
/* get polling slot */
#define TPUART_GET_POLLING_SLOT _IOR (TPUART_MAGIC, 12, unsigned char)
/* set polling address */
#define TPUART_SET_POLLING_ADDR _IOW (TPUART_MAGIC, 13, unsigned short)
/* get polling address */
#define TPUART_GET_POLLING_ADDR _IOR (TPUART_MAGIC, 14, unsigned short)
/* set polling response */
#define TPUART_SET_POLLING_RESP _IOW (TPUART_MAGIC, 15, unsigned char)
/* get polling response */
#define TPUART_GET_POLLING_RESP _IOR (TPUART_MAGIC, 16, unsigned char)
/* get tpuart_state */
#define TPUART_GET_STATE _IOR (TPUART_MAGIC, 21, unsigned char)
/* get last write state */
#define TPUART_GET_WRITE_STATUS _IOR (TPUART_MAGIC, 22, int)


#endif /* __TPUART_MAIN */
