/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: transmit.c -- transmit routine
 *
 *    $Id: transmit.c,v 1.2 2001/05/23 11:33:49 rst Exp $
 *
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    07-Aug-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <asm/io.h>
#include "tpuart_main.h"


/* bottom half for transmit */
void transmit (void *arg)
{
	tpuart_t *dev = arg;
	unsigned char padbyte = U_L_DATA_START;
	unsigned short length;
	unsigned short pos = 0;
	unsigned long flags;
	unsigned char check_octet = 0;   

	if (test_bit (SEND_AND_WAIT, &dev->tx.flags)) {
		/* if we can't send now, signal sending request */
		PDEBUG ("Hell, can't send...\n");
		set_bit (SEND_REQU, &dev->tx.flags);
	} else {
		/* sending in int_handler must be complete at this point! (that's what SEND_AND_WAIT for) */
		dev->tx.write = dev->tx.buffer;
     
		if (!deque_get_irq (&dev->tx.temp, &dev->tx.tel_buffer)) {
			/* appearently nothing to read */
			PDEBUG ("Tried to read empty queue!\n");
			return;
		}
		PDEBUG ("Sending message!\n");
		clear_bit (SEND_REQU, &dev->tx.flags);
		set_bit (SEND_AND_WAIT, &dev->tx.flags);       //set until confirm or error is received 
		set_bit (DID_SEND, &dev->tx.flags);            //necessary to check if we did sent when we receive the message from TPUART

	        (dev->tx.temp.data_length)++;  //increment length to include the checksum
		                               //in the new version the checksum is calculated by the driver

		length = dev->tx.temp.data_length;
		if (length > 64) /* sanity check */
			length = 64;

		/* create message which can be send to TPUART (padbyte + messagebyte) */
		for (; length > 1; length--) {
			PDEBUG ("Queuing padbyte: 0x%x msgbyte: 0x%x\n", padbyte, 
				dev->tx.temp.data[pos]);
			*dev->tx.write++ = padbyte++;
			*dev->tx.write++ = dev->tx.temp.data[pos];
			
			check_octet ^= (dev->tx.temp.data[pos++]);        //calculate checksum		

		}
		check_octet = ~check_octet;     //invert checksum for odd parity
		PDEBUG ("Sending last padbyte: 0x%x check octet: 0x%x\n", (padbyte | 0x40) & 0x7F,
			check_octet);
		*dev->tx.write++ = (padbyte | 0x40) & 0x7F;  /* end code is 0x4i (i = index) */
         	*dev->tx.write =check_octet;                /* add checksum as last byte */
	
		dev->tx.end = dev->tx.write;
		dev->tx.read = dev->tx.buffer;
	
			
		if(test_bit(DEAD, &dev->flags)){         //don't start new message if we are dead
		  return;
		}
	

		/* write out first 2 bytes to provocate tx interrupt */
		set_bit (SEND_IN_PROGRESS, &dev->tx.flags);                //flag set until last byte is transmitted via serial
		
		spin_lock_irqsave(&dev->lock, flags);
				
		outb (*dev->tx.read++, dev->port);
		outb (*dev->tx.read++, dev->port);
		
		spin_unlock_irqrestore(&dev->lock, flags);
		
	}
}
