/* 
 *    tpuart -- A TP-UART driver 
 *
 *
 *    File: utility.c -- general utility functions
 *
 *    $Id: utility.c,v 1.3 2001/05/23 11:33:49 rst Exp $
 *
 *
 *    Copyright (C) 2000 Raffael Stocker <raffael.stocker@stud.fh-deggendorf.de> 
 *                       University of applied sciences Deggendorf
 *
 *    July 2004   Porting driver to kernel 2.6
 *                Reinhold Buchinger <e0125124@student.tuwien.ac.at>
 *                University of Technology Vienna
 *
 *    This program is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU General Public License as
 *    published by the Free Software Foundation; either version 2 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
 *    02111-1307, USA.
 *
 *
 *    31-Jul-2000: 0.01  Started Coding
 *
 *    Last change:
 *
 *    $Date: 2001/05/23 11:33:49 $ $Author: rst $
 *
 *    $Date: 2004/07/23 08:00:00 $ $Author: rb  $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/time.h>
#include <asm/io.h>
#include "tpuart_main.h"

int is_my_group (unsigned short, tpuart_t*);
int is_my_addr (unsigned short, tpuart_t*);

/*
 * is_timeout () returns 1 if there are more than TEL_TIMEOUT ms
 * between the two arguments.
 * t2 MUST be the later time.
 */
int is_timeout (struct timeval t1, struct timeval t2)
{
	suseconds_t udiff;
	time_t sdiff;

	if (t1.tv_usec > t2.tv_usec)
		udiff = 1000000 - t1.tv_usec + t2.tv_usec;
	else
		udiff = t2.tv_usec - t1.tv_usec;

	sdiff = t2.tv_sec - t1.tv_sec;

	if (sdiff > 1) {
		PDEBUGG ("TIMEOUT: t2.tv_sec = %u, t1.tv_sec = %u, sdiff = %u\n", t2.tv_sec, t1.tv_sec, sdiff);
		return 1;
	}

	if (udiff >= TEL_TIMEOUT) {
		PDEBUGG ("TIMEOUT: t2.tv_sec = %u, t2.tv_usec = %u, "
			"t1.tv_sec = %u, t1.tv_usec = %u, udiff = %u\n", t2.tv_sec, t2.tv_usec, 
			t1.tv_sec, t1.tv_usec, udiff);
		return 1;
	}
	return 0;       /* no timeout */
}

/*
 * is_addressed () returns 1 if
 * we are the destination.
 */
int is_addressed (tpuart_t *dev)
{
	unsigned short addr = 0;
  
	addr =  *(dev->rx.buffer + ((dev->rx.first+DEST_ADDR_HIGH*2)  - dev->rx.buffer) % (2*BUFSIZE)) << 8;
	addr |= *(dev->rx.buffer + ((dev->rx.first+DEST_ADDR_LOW*2)  - dev->rx.buffer) % (2*BUFSIZE));      

	PDEBUGG ("address: 0x%u\n", addr);
	if (*(dev->rx.buffer + ((dev->rx.first+NPCI*2)  - dev->rx.buffer) % (2*BUFSIZE)) & GROUP_ADDR) {
		PDEBUGG ("Group-addressed!!!\n");
		return is_my_group (addr, dev);
	} else
		return is_my_addr (addr, dev);
}

/*
 * is_my_group () returns true if we are a
 * member of the asked group.
 */
int is_my_group (unsigned short addr, tpuart_t *dev)
{
  unsigned long flags;
  spin_lock_irqsave(&dev->lock, flags);
  return dev->group_addr[addr>>3] & (1 << (0x7 & addr));
  spin_unlock_irqrestore(&dev->lock,flags);
}

/*
 * is_my_addr () returns true if we have
 * the asked physical address.
 */
int is_my_addr (unsigned short addr, tpuart_t *dev)
{
  unsigned long flags;
  spin_lock_irqsave(&dev->lock, flags);
  return dev->phys_addr[addr>>3] & (1 << (0x7 & addr));
  spin_unlock_irqrestore(&dev->lock,flags);

}


/*
 * This is a support routine for tpuart_open.
 * It initializes the serial port
 */
int ser_init (tpuart_t *dev)
{
        int port = dev->port;
	/* Part of uart autodetection. Stolen from serial.c */
	unsigned long flags;
	unsigned char scratch, scratch2, scratch3;

	spin_lock_irqsave(&dev->lock,flags);

	PDEBUGG ("init port 0x%02x\n", port);
	scratch = inb (port + IER);
	outb (0, port + IER);
#ifdef __i386__
	outb (0xff, 0x080);
#endif
	scratch2 = inb (port + IER);
	outb (0x0F, port + IER);
#ifdef __i386__
	outb (0, 0x080);
#endif
	scratch3 = inb (port + IER);
	outb (scratch, port + IER);
	if (scratch2 || scratch3 != 0x0F) {
		PDEBUGG("%02x: no uart detected "
		       "(%02x, %02x)\n", port, 
		       scratch2, scratch3);
		spin_unlock_irqrestore(&dev->lock, flags);
		return (0);
	}

	outb (WLEN|PSEL|DLAB, port+LCR); /* set DLAB bit (used for baud
					    rate selection) */
	outb (6, port+DLLB);   /* 115200/19200 = 6 */
	outb (0, port+DLHB);   
	outb (WLEN|PSEL, port+LCR); /* clear DLAB bit */
	outb (FDTR|OUT2, port+MCR); /* activate DTR-line and master-interrupt active */
	while (inb (port+LSR) & DRDY)
		inb (port); /* dummy read all bytes */
	outb (0, port+IER);
	outb (FEN, port+FCR);  /* switch on FIFO, 1 Byte Interrupt trigger level */
	inb (port); /* dummy-read port */
	outb (RIE|TIE|LSIE, port+IER); /* actually switch ints on */

	spin_unlock_irqrestore(&dev->lock, flags);
	return (1);
}

